#!/usr/bin/env python3
"""
Search hotels by city or coordinates via Amadeus API.
"""

import argparse
import json
import sys
import time
from typing import Optional

import requests

from auth import get_auth_header, get_base_url


def make_request(url: str, params: dict, retries: int = 3) -> dict:
    """Make API request with retry logic for rate limits."""
    headers = get_auth_header()
    
    for attempt in range(retries):
        response = requests.get(url, headers=headers, params=params, timeout=30)
        
        if response.status_code == 429:
            wait = 2 ** attempt
            print(f"Rate limited, waiting {wait}s...", file=sys.stderr)
            time.sleep(wait)
            continue
        
        if response.status_code == 401:
            raise EnvironmentError(
                "Authentication failed. Check AMADEUS_API_KEY and AMADEUS_API_SECRET."
            )
        
        if response.status_code == 400:
            error = response.json().get("errors", [{}])[0]
            raise ValueError(f"Bad request: {error.get('detail', response.text)}")
        
        response.raise_for_status()
        return response.json()
    
    raise Exception("Max retries exceeded due to rate limiting")


def search_by_city(
    city_code: str,
    amenities: Optional[list] = None,
    ratings: Optional[list] = None,
) -> list:
    """Search hotels by city code."""
    base_url = get_base_url()
    params = {"cityCode": city_code.upper()}
    
    if amenities:
        params["amenities"] = ",".join(amenities)
    if ratings:
        params["ratings"] = ",".join(str(r) for r in ratings)
    
    data = make_request(
        f"{base_url}/v1/reference-data/locations/hotels/by-city",
        params,
    )
    
    return data.get("data", [])


def search_by_geocode(
    latitude: float,
    longitude: float,
    radius: int = 5,
    amenities: Optional[list] = None,
    ratings: Optional[list] = None,
) -> list:
    """Search hotels by coordinates."""
    base_url = get_base_url()
    params = {
        "latitude": latitude,
        "longitude": longitude,
        "radius": radius,
        "radiusUnit": "KM",
    }
    
    if amenities:
        params["amenities"] = ",".join(amenities)
    if ratings:
        params["ratings"] = ",".join(str(r) for r in ratings)
    
    data = make_request(
        f"{base_url}/v1/reference-data/locations/hotels/by-geocode",
        params,
    )
    
    return data.get("data", [])


def format_rating(rating: Optional[int]) -> str:
    """Format star rating."""
    if not rating:
        return ""
    return "★" * rating


def format_human(hotels: list) -> str:
    """Format hotels for human reading."""
    if not hotels:
        return "No hotels found."
    
    lines = []
    for hotel in hotels[:20]:  # Limit output
        name = hotel.get("name", "Unknown")
        hotel_id = hotel.get("hotelId", "")
        rating = hotel.get("rating")
        
        # Address
        addr = hotel.get("address", {})
        city = addr.get("cityName", "")
        
        # Amenities
        amenities = hotel.get("amenities", [])
        amenity_str = ", ".join(amenities[:5]) if amenities else ""
        
        lines.append(f"🏨 {name} {format_rating(rating)} ({hotel_id})")
        if city:
            lines.append(f"   📍 {city}")
        if amenity_str:
            lines.append(f"   ✨ {amenity_str}")
        lines.append("")
    
    if len(hotels) > 20:
        lines.append(f"... and {len(hotels) - 20} more hotels")
    
    return "\n".join(lines)


def main():
    parser = argparse.ArgumentParser(description="Search hotels via Amadeus API")
    
    # Location options
    parser.add_argument("--city", help="City code (IATA), e.g., PAR, NYC, TYO")
    parser.add_argument("--lat", type=float, help="Latitude (use with --lon)")
    parser.add_argument("--lon", type=float, help="Longitude (use with --lat)")
    parser.add_argument("--radius", type=int, default=5, help="Search radius in km (default: 5)")
    
    # Filters
    parser.add_argument("--amenities", help="Comma-separated amenities: WIFI,POOL,SPA")
    parser.add_argument("--ratings", help="Comma-separated star ratings: 4,5")
    
    # Output
    parser.add_argument("--format", choices=["json", "human"], default="json",
                        help="Output format (default: json)")
    
    args = parser.parse_args()
    
    # Validate input
    if not args.city and not (args.lat and args.lon):
        parser.error("Either --city or both --lat and --lon required")
    
    # Parse filters
    amenities = args.amenities.split(",") if args.amenities else None
    ratings = [int(r) for r in args.ratings.split(",")] if args.ratings else None
    
    try:
        if args.city:
            hotels = search_by_city(args.city, amenities, ratings)
        else:
            hotels = search_by_geocode(args.lat, args.lon, args.radius, amenities, ratings)
        
        if args.format == "human":
            print(format_human(hotels))
        else:
            print(json.dumps(hotels, indent=2))
    
    except Exception as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)


if __name__ == "__main__":
    main()
