#!/usr/bin/env python3
"""
Get hotel pricing and availability via Amadeus API.
"""

import argparse
import json
import sys
import time
from datetime import date
from typing import Optional

import requests

from auth import get_auth_header, get_base_url


def make_request(url: str, params: dict, retries: int = 3) -> dict:
    """Make API request with retry logic for rate limits."""
    headers = get_auth_header()
    
    for attempt in range(retries):
        response = requests.get(url, headers=headers, params=params, timeout=30)
        
        if response.status_code == 429:
            wait = 2 ** attempt
            print(f"Rate limited, waiting {wait}s...", file=sys.stderr)
            time.sleep(wait)
            continue
        
        if response.status_code == 401:
            raise EnvironmentError(
                "Authentication failed. Check AMADEUS_API_KEY and AMADEUS_API_SECRET."
            )
        
        if response.status_code == 400:
            error = response.json().get("errors", [{}])[0]
            raise ValueError(f"Bad request: {error.get('detail', response.text)}")
        
        response.raise_for_status()
        return response.json()
    
    raise Exception("Max retries exceeded due to rate limiting")


def get_offers(
    hotel_ids: list[str],
    check_in: str,
    check_out: str,
    adults: int = 2,
    rooms: int = 1,
    currency: str = "USD",
) -> list:
    """Get pricing offers for hotels."""
    base_url = get_base_url()
    
    params = {
        "hotelIds": ",".join(hotel_ids),
        "adults": adults,
        "roomQuantity": rooms,
        "checkInDate": check_in,
        "checkOutDate": check_out,
        "currency": currency,
    }
    
    data = make_request(f"{base_url}/v3/shopping/hotel-offers", params)
    return data.get("data", [])


def calculate_nights(check_in: str, check_out: str) -> int:
    """Calculate number of nights."""
    from datetime import datetime
    ci = datetime.strptime(check_in, "%Y-%m-%d")
    co = datetime.strptime(check_out, "%Y-%m-%d")
    return (co - ci).days


def format_price(price: dict, nights: int) -> str:
    """Format price information."""
    total = float(price.get("total", 0))
    currency = price.get("currency", "USD")
    per_night = total / nights if nights > 0 else total
    
    # Currency symbols
    symbols = {"USD": "$", "EUR": "€", "GBP": "£", "JPY": "¥"}
    symbol = symbols.get(currency, currency + " ")
    
    return f"{symbol}{per_night:.0f}/night ({symbol}{total:.0f} total for {nights} nights)"


def format_cancellation(policies: list) -> str:
    """Format cancellation policy."""
    if not policies:
        return "Policy not specified"
    
    policy = policies[0]
    policy_type = policy.get("type", "")
    
    if policy_type == "NON_REFUNDABLE":
        return "❌ Non-refundable"
    elif policy_type == "REFUNDABLE":
        deadline = policy.get("deadline", "")
        if deadline:
            return f"✅ Free cancellation until {deadline[:10]}"
        return "✅ Refundable"
    
    return policy_type


def format_board(board: Optional[str]) -> str:
    """Format board type."""
    boards = {
        "ROOM_ONLY": "Room only",
        "BREAKFAST": "🍳 Breakfast included",
        "HALF_BOARD": "🍳🍽️ Half board (breakfast + dinner)",
        "FULL_BOARD": "🍳🍽️🍽️ Full board (all meals)",
        "ALL_INCLUSIVE": "🌟 All-inclusive",
    }
    return boards.get(board, board or "Room only")


def format_human(hotels_with_offers: list, check_in: str, check_out: str) -> str:
    """Format offers for human reading."""
    if not hotels_with_offers:
        return "No offers found for the specified dates."
    
    nights = calculate_nights(check_in, check_out)
    lines = []
    
    for hotel_data in hotels_with_offers:
        hotel = hotel_data.get("hotel", {})
        offers = hotel_data.get("offers", [])
        
        name = hotel.get("name", "Unknown Hotel")
        lines.append(f"🏨 {name}")
        
        for i, offer in enumerate(offers[:3], 1):  # Limit to 3 offers per hotel
            room = offer.get("room", {})
            room_type = room.get("typeEstimated", {})
            room_name = room_type.get("category", "Standard Room")
            beds = room_type.get("beds", 1)
            bed_type = room_type.get("bedType", "")
            
            price = offer.get("price", {})
            price_str = format_price(price, nights)
            
            policies = offer.get("policies", {}).get("cancellation", {})
            cancel_str = format_cancellation([policies] if policies else [])
            
            board = offer.get("boardType")
            board_str = format_board(board)
            
            offer_id = offer.get("id", "")
            
            lines.append(f"   🛏️ {room_name} ({beds} {bed_type})")
            lines.append(f"   💰 {price_str}")
            lines.append(f"   {board_str}")
            lines.append(f"   {cancel_str}")
            lines.append(f"   📋 Offer ID: {offer_id}")
            lines.append("")
        
        lines.append("")
    
    return "\n".join(lines)


def main():
    parser = argparse.ArgumentParser(description="Get hotel offers via Amadeus API")
    
    parser.add_argument("--hotels", required=True,
                        help="Comma-separated hotel IDs from search")
    parser.add_argument("--checkin", required=True,
                        help="Check-in date (YYYY-MM-DD)")
    parser.add_argument("--checkout", required=True,
                        help="Check-out date (YYYY-MM-DD)")
    parser.add_argument("--adults", type=int, default=2,
                        help="Number of adults (default: 2)")
    parser.add_argument("--rooms", type=int, default=1,
                        help="Number of rooms (default: 1)")
    parser.add_argument("--currency", default="USD",
                        help="Currency code (default: USD)")
    parser.add_argument("--format", choices=["json", "human"], default="json",
                        help="Output format (default: json)")
    
    args = parser.parse_args()
    
    hotel_ids = [h.strip() for h in args.hotels.split(",")]
    
    try:
        offers = get_offers(
            hotel_ids,
            args.checkin,
            args.checkout,
            args.adults,
            args.rooms,
            args.currency,
        )
        
        if args.format == "human":
            print(format_human(offers, args.checkin, args.checkout))
        else:
            print(json.dumps(offers, indent=2))
    
    except Exception as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)


if __name__ == "__main__":
    main()
