#!/usr/bin/env python3
"""
Get offer details and hotel ratings via Amadeus API.
"""

import argparse
import json
import sys
import time
from typing import Optional

import requests

from auth import get_auth_header, get_base_url


def make_request(url: str, params: Optional[dict] = None, retries: int = 3) -> dict:
    """Make API request with retry logic for rate limits."""
    headers = get_auth_header()
    
    for attempt in range(retries):
        response = requests.get(url, headers=headers, params=params or {}, timeout=30)
        
        if response.status_code == 429:
            wait = 2 ** attempt
            print(f"Rate limited, waiting {wait}s...", file=sys.stderr)
            time.sleep(wait)
            continue
        
        if response.status_code == 401:
            raise EnvironmentError(
                "Authentication failed. Check AMADEUS_API_KEY and AMADEUS_API_SECRET."
            )
        
        if response.status_code == 400:
            error = response.json().get("errors", [{}])[0]
            raise ValueError(f"Bad request: {error.get('detail', response.text)}")
        
        if response.status_code == 404:
            return {"data": None}
        
        response.raise_for_status()
        return response.json()
    
    raise Exception("Max retries exceeded due to rate limiting")


def get_offer_details(offer_id: str) -> dict:
    """Get full details for a specific offer."""
    base_url = get_base_url()
    data = make_request(f"{base_url}/v3/shopping/hotel-offers/{offer_id}")
    return data.get("data", {})


def get_hotel_info(hotel_id: str) -> dict:
    """Get static hotel information."""
    base_url = get_base_url()
    params = {"hotelIds": hotel_id}
    data = make_request(
        f"{base_url}/v1/reference-data/locations/hotels/by-hotels",
        params,
    )
    hotels = data.get("data", [])
    return hotels[0] if hotels else {}


def get_hotel_ratings(hotel_id: str) -> dict:
    """Get hotel sentiment/ratings."""
    base_url = get_base_url()
    params = {"hotelIds": hotel_id}
    data = make_request(
        f"{base_url}/v2/e-reputation/hotel-sentiments",
        params,
    )
    ratings = data.get("data", [])
    return ratings[0] if ratings else {}


def format_rating_bar(score: int, max_score: int = 100) -> str:
    """Create visual rating bar."""
    filled = int(score / max_score * 10)
    return "█" * filled + "░" * (10 - filled)


def format_ratings_human(ratings: dict) -> str:
    """Format ratings for human reading."""
    if not ratings:
        return "No ratings available."
    
    lines = []
    overall = ratings.get("overallRating", 0)
    lines.append(f"📊 Overall Rating: {overall}/100 {format_rating_bar(overall)}")
    lines.append("")
    
    # Category scores
    sentiments = ratings.get("sentiments", [])
    if sentiments:
        lines.append("Category Scores:")
        for sentiment in sorted(sentiments, key=lambda x: -x.get("score", 0)):
            category = sentiment.get("category", "Unknown")
            score = sentiment.get("score", 0)
            # Make category name readable
            cat_name = category.replace("_", " ").title()
            lines.append(f"   {cat_name}: {score}/100 {format_rating_bar(score)}")
    
    num_reviews = ratings.get("numberOfReviews", 0)
    if num_reviews:
        lines.append(f"\n📝 Based on {num_reviews:,} reviews")
    
    return "\n".join(lines)


def format_offer_human(offer: dict) -> str:
    """Format offer details for human reading."""
    if not offer:
        return "Offer not found or expired."
    
    lines = []
    
    hotel = offer.get("hotel", {})
    lines.append(f"🏨 {hotel.get('name', 'Unknown Hotel')}")
    
    # Room details
    offers = offer.get("offers", [offer]) if "offers" not in offer else offer.get("offers", [])
    
    for o in offers[:1]:  # Usually just one offer in details
        room = o.get("room", {})
        desc = room.get("description", {})
        desc_text = desc.get("text", "")
        
        if desc_text:
            lines.append(f"\n📝 Room Description:")
            lines.append(f"   {desc_text[:500]}")
        
        # Price
        price = o.get("price", {})
        total = price.get("total", "N/A")
        currency = price.get("currency", "")
        lines.append(f"\n💰 Total Price: {currency} {total}")
        
        # Payment info
        payment = o.get("policies", {}).get("paymentType", "")
        if payment:
            lines.append(f"💳 Payment: {payment}")
        
        # Cancellation
        cancel = o.get("policies", {}).get("cancellation", {})
        if cancel:
            cancel_type = cancel.get("type", "")
            deadline = cancel.get("deadline", "")
            amount = cancel.get("amount", {})
            
            if cancel_type == "NON_REFUNDABLE":
                lines.append("❌ Non-refundable")
            else:
                if deadline:
                    lines.append(f"✅ Free cancellation until {deadline[:10]}")
                if amount:
                    lines.append(f"   Cancellation fee: {amount.get('currency', '')} {amount.get('value', '')}")
        
        # Check-in/out
        checkin = o.get("checkInDate", "")
        checkout = o.get("checkOutDate", "")
        if checkin and checkout:
            lines.append(f"\n📅 {checkin} → {checkout}")
        
        # Guests
        guests = o.get("guests", {})
        adults = guests.get("adults", 0)
        if adults:
            lines.append(f"👥 {adults} adult(s)")
    
    return "\n".join(lines)


def format_hotel_human(hotel: dict, ratings: Optional[dict] = None) -> str:
    """Format hotel info for human reading."""
    if not hotel:
        return "Hotel not found."
    
    lines = []
    
    name = hotel.get("name", "Unknown")
    rating = hotel.get("rating", "")
    stars = "★" * int(rating) if rating else ""
    
    lines.append(f"🏨 {name} {stars}")
    
    # Address
    addr = hotel.get("address", {})
    if addr:
        street = addr.get("lines", [""])[0] if addr.get("lines") else ""
        city = addr.get("cityName", "")
        country = addr.get("countryCode", "")
        lines.append(f"📍 {street}, {city}, {country}")
    
    # Contact
    contact = hotel.get("contact", {})
    phone = contact.get("phone", "")
    email = contact.get("email", "")
    if phone:
        lines.append(f"📞 {phone}")
    if email:
        lines.append(f"📧 {email}")
    
    # Amenities
    amenities = hotel.get("amenities", [])
    if amenities:
        lines.append(f"\n✨ Amenities: {', '.join(amenities[:10])}")
    
    # Add ratings if available
    if ratings:
        lines.append("\n" + format_ratings_human(ratings))
    
    return "\n".join(lines)


def main():
    parser = argparse.ArgumentParser(description="Get hotel/offer details via Amadeus API")
    
    parser.add_argument("--offer-id", help="Get details for specific offer")
    parser.add_argument("--hotel-id", help="Get hotel info (use with --ratings)")
    parser.add_argument("--ratings", action="store_true",
                        help="Include ratings/sentiment data")
    parser.add_argument("--format", choices=["json", "human"], default="json",
                        help="Output format (default: json)")
    
    args = parser.parse_args()
    
    if not args.offer_id and not args.hotel_id:
        parser.error("Either --offer-id or --hotel-id required")
    
    try:
        result = {}
        
        if args.offer_id:
            result["offer"] = get_offer_details(args.offer_id)
        
        if args.hotel_id:
            result["hotel"] = get_hotel_info(args.hotel_id)
            if args.ratings:
                result["ratings"] = get_hotel_ratings(args.hotel_id)
        
        if args.format == "human":
            if args.offer_id:
                print(format_offer_human(result.get("offer", {})))
            else:
                print(format_hotel_human(
                    result.get("hotel", {}),
                    result.get("ratings") if args.ratings else None,
                ))
        else:
            print(json.dumps(result, indent=2))
    
    except Exception as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)


if __name__ == "__main__":
    main()
