#!/usr/bin/env python3
"""
Amadeus OAuth 2.0 token management with caching.
"""

import json
import os
import time
from pathlib import Path
from typing import Optional

import requests

# State directory for token cache
STATE_DIR = Path(__file__).parent.parent / "state"
TOKEN_FILE = STATE_DIR / "token.json"


def get_base_url() -> str:
    """Get API base URL based on environment."""
    env = os.environ.get("AMADEUS_ENV", "test").lower()
    if env == "production":
        return "https://api.amadeus.com"
    return "https://test.api.amadeus.com"


def get_credentials() -> tuple[str, str]:
    """Get API credentials from environment."""
    api_key = os.environ.get("AMADEUS_API_KEY")
    api_secret = os.environ.get("AMADEUS_API_SECRET")
    
    if not api_key or not api_secret:
        raise EnvironmentError(
            "Missing credentials. Set AMADEUS_API_KEY and AMADEUS_API_SECRET."
        )
    
    return api_key, api_secret


def load_cached_token() -> Optional[dict]:
    """Load token from cache if valid."""
    if not TOKEN_FILE.exists():
        return None
    
    try:
        with open(TOKEN_FILE) as f:
            data = json.load(f)
        
        # Check if expired (with 60s buffer)
        if data.get("expires_at", 0) > time.time() + 60:
            return data
    except (json.JSONDecodeError, KeyError):
        pass
    
    return None


def save_token(token_data: dict) -> None:
    """Save token to cache."""
    STATE_DIR.mkdir(parents=True, exist_ok=True)
    with open(TOKEN_FILE, "w") as f:
        json.dump(token_data, f, indent=2)


def fetch_new_token() -> dict:
    """Fetch new OAuth token from Amadeus."""
    api_key, api_secret = get_credentials()
    base_url = get_base_url()
    
    response = requests.post(
        f"{base_url}/v1/security/oauth2/token",
        headers={"Content-Type": "application/x-www-form-urlencoded"},
        data={
            "grant_type": "client_credentials",
            "client_id": api_key,
            "client_secret": api_secret,
        },
        timeout=30,
    )
    
    if response.status_code == 401:
        raise EnvironmentError(
            "Authentication failed. Check AMADEUS_API_KEY and AMADEUS_API_SECRET."
        )
    
    response.raise_for_status()
    data = response.json()
    
    # Add expiry timestamp
    expires_in = data.get("expires_in", 1799)  # Default 30 min
    token_data = {
        "access_token": data["access_token"],
        "token_type": data.get("token_type", "Bearer"),
        "expires_in": expires_in,
        "expires_at": time.time() + expires_in,
    }
    
    save_token(token_data)
    return token_data


def get_token() -> str:
    """Get valid access token (from cache or fresh)."""
    cached = load_cached_token()
    if cached:
        return cached["access_token"]
    
    token_data = fetch_new_token()
    return token_data["access_token"]


def get_auth_header() -> dict:
    """Get Authorization header for API requests."""
    token = get_token()
    return {"Authorization": f"Bearer {token}"}


if __name__ == "__main__":
    # Test token retrieval
    try:
        token = get_token()
        print(f"Token obtained: {token[:20]}...")
        print(f"Environment: {os.environ.get('AMADEUS_ENV', 'test')}")
        print(f"Base URL: {get_base_url()}")
    except Exception as e:
        print(f"Error: {e}")
        exit(1)
