#!/bin/bash
#
# AIUSD Skill - Self-Extracting Shell Installer
#
# This file contains the complete AIUSD trading skill package.
# Simply run: bash aiusd-skill-installer.sh
#
# Generated: 2026-02-06T09:37:15.352Z
# Version: 1.0.0
#

set -euo pipefail

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
MAGENTA='\033[0;35m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

log_info() { echo -e "${BLUE}ℹ️${NC} $1"; }
log_success() { echo -e "${GREEN}✅${NC} $1"; }
log_warning() { echo -e "${YELLOW}⚠️${NC} $1"; }
log_error() { echo -e "${RED}❌${NC} $1"; }

main() {
    echo -e "${MAGENTA}🚀 AIUSD Skill Installer${NC}"
    echo -e "${MAGENTA}========================${NC}"
    echo -e "${BLUE}📦 Version: 1.0.0${NC}"
    echo ""

    # Check dependencies
    if ! command -v node >/dev/null 2>&1; then
        log_error "Node.js not found. Please install Node.js 18+ from https://nodejs.org"
        exit 1
    fi

    if ! command -v npm >/dev/null 2>&1; then
        log_error "npm not found. Please install npm"
        exit 1
    fi

    NODE_VERSION=$(node --version | cut -d'v' -f2)
    REQUIRED_MAJOR=18
    CURRENT_MAJOR=$(echo "$NODE_VERSION" | cut -d'.' -f1)

    if [[ $CURRENT_MAJOR -lt $REQUIRED_MAJOR ]]; then
        log_error "Node.js $REQUIRED_MAJOR+ required, but found v$NODE_VERSION"
        exit 1
    fi

    log_success "Node.js v$NODE_VERSION found"

    # Extract to current directory
    INSTALL_DIR=$(pwd)
    SKILL_DIR="$INSTALL_DIR/aiusd-skill"

    log_info "Installing to: $SKILL_DIR"

    # Create skill directory
    if [[ -d "$SKILL_DIR" ]]; then
        log_warning "Removing existing installation..."
        rm -rf "$SKILL_DIR"
    fi
    mkdir -p "$SKILL_DIR"

    # Extract package data
    log_info "Extracting skill package..."

    # Find the start of the base64 data
    ARCHIVE_START=$(awk '/^__ARCHIVE_START__$/{print NR+1; exit 0; }' "$0")

    # Extract and decode the archive
    tail -n +$ARCHIVE_START "$0" | base64 -d | tar -xzf - -C "$SKILL_DIR"

    # Install dependencies
    log_info "Installing dependencies..."
    cd "$SKILL_DIR"

    if npm install >/dev/null 2>&1; then
        log_success "Dependencies installed successfully"
    else
        log_warning "Failed to install dependencies automatically"
        log_info "Please run manually: cd aiusd-skill && npm install"
    fi

    echo ""
    log_success "AIUSD Skill installed successfully!"
    echo ""
    echo -e "${YELLOW}🚀 Next Steps:${NC}"
    echo -e "${BLUE}1. cd aiusd-skill${NC}"
    echo -e "${BLUE}2. npm run setup${NC}"
    echo ""
    echo -e "${CYAN}💡 Usage Examples:${NC}"
    echo -e "${BLUE}• Check balance: npm start -- balances${NC}"
    echo -e "${BLUE}• List tools: npm start -- tools${NC}"
    echo -e "${BLUE}• Get help: npm start -- --help${NC}"
    echo ""

    exit 0
}

# Run main function unless sourced
if [[ "${BASH_SOURCE[0]}" == "${0}" ]]; then
    main "$@"
fi

# Archive marker - do not remove this line
__ARCHIVE_START__
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