#!/bin/bash
#
# AIUSD Skill - Self-Extracting Shell Installer
#
# This file contains the complete AIUSD trading skill package.
# Simply run: bash aiusd-skill-installer.sh
#
# Generated: 2026-02-11T01:42:02.574Z
# Version: 1.0.0
#

set -euo pipefail

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
MAGENTA='\033[0;35m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

log_info() { echo -e "${BLUE}ℹ️${NC} $1"; }
log_success() { echo -e "${GREEN}✅${NC} $1"; }
log_warning() { echo -e "${YELLOW}⚠️${NC} $1"; }
log_error() { echo -e "${RED}❌${NC} $1"; }

main() {
    echo -e "${MAGENTA}🚀 AIUSD Skill Installer${NC}"
    echo -e "${MAGENTA}========================${NC}"
    echo -e "${BLUE}📦 Version: 1.0.0${NC}"
    echo ""

    # Check dependencies
    if ! command -v node >/dev/null 2>&1; then
        log_error "Node.js not found. Please install Node.js 18+ from https://nodejs.org"
        exit 1
    fi

    if ! command -v npm >/dev/null 2>&1; then
        log_error "npm not found. Please install npm"
        exit 1
    fi

    NODE_VERSION=$(node --version | cut -d'v' -f2)
    REQUIRED_MAJOR=18
    CURRENT_MAJOR=$(echo "$NODE_VERSION" | cut -d'.' -f1)

    if [[ $CURRENT_MAJOR -lt $REQUIRED_MAJOR ]]; then
        log_error "Node.js $REQUIRED_MAJOR+ required, but found v$NODE_VERSION"
        exit 1
    fi

    log_success "Node.js v$NODE_VERSION found"

    # Extract to current directory
    INSTALL_DIR=$(pwd)
    SKILL_DIR="$INSTALL_DIR/aiusd-skill"

    log_info "Installing to: $SKILL_DIR"

    # Create skill directory
    if [[ -d "$SKILL_DIR" ]]; then
        log_warning "Removing existing installation..."
        rm -rf "$SKILL_DIR"
    fi
    mkdir -p "$SKILL_DIR"

    # Extract package data
    log_info "Extracting skill package..."

    # Find the start of the base64 data
    ARCHIVE_START=$(awk '/^__ARCHIVE_START__$/{print NR+1; exit 0; }' "$0")

    # Extract and decode the archive
    tail -n +$ARCHIVE_START "$0" | base64 -d | tar -xzf - -C "$SKILL_DIR"

    # Install dependencies
    log_info "Installing dependencies..."
    cd "$SKILL_DIR"

    if npm install >/dev/null 2>&1; then
        log_success "Dependencies installed successfully"
    else
        log_warning "Failed to install dependencies automatically"
        log_info "Please run manually: cd aiusd-skill && npm install"
    fi

    echo ""
    log_success "AIUSD Skill installed successfully!"
    echo ""
    echo -e "${YELLOW}🚀 Next Steps:${NC}"
    echo -e "${BLUE}1. cd aiusd-skill${NC}"
    echo -e "${BLUE}2. npm run setup${NC}"
    echo ""
    echo -e "${CYAN}💡 Usage Examples:${NC}"
    echo -e "${BLUE}• Check balance: npm start -- balances${NC}"
    echo -e "${BLUE}• List tools: npm start -- tools${NC}"
    echo -e "${BLUE}• Get help: npm start -- --help${NC}"
    echo ""

    exit 0
}

# Run main function unless sourced
if [[ "${BASH_SOURCE[0]}" == "${0}" ]]; then
    main "$@"
fi

# Archive marker - do not remove this line
__ARCHIVE_START__
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