---
name: aisp
version: 1.0.2
description: Enables AI agents to interact with AISP (Agent Inference Sharing Protocol) for renting or providing DIEM API capacity. Use when working with diem-marketplace, Venice API keys, USDC escrow, listings, rentals, or when the user wants to rent inference capacity or list API keys.
metadata: {"openclaw":{"homepage":"https://github.com/DaveO280/Diem-Marketplace-V2-","emoji":"⚡","requires":{"env":["BACKEND_URL"]}}}
---

# AISP Agent Skill

Agent Inference Sharing Protocol (AISP) lets agents rent idle DIEM/Venice API capacity via USDC escrow. Providers list capped API keys; agents fund and receive keys automatically.

## Architecture

```
Agent: fund() → Backend sees Funded event → Key released → Agent uses Venice API
Provider: list() → Agent funds → Term expires → settle() → Provider paid (99%, 1% fee)
```

## Agent Workflow (Renting)

1. **Listings** from backend: `GET /api/listings`
2. **Approve USDC** if needed (contract spends on `fund`)
3. **Fund** on-chain: `contract.fund(listingId, termDays, diemAmount)` → returns `rentalId`
4. **Get key**: `POST /api/key/{rentalId}` with signed message `diem-marketplace:get-key:{rentalId}:{timestamp}`
5. Use `apiKey` with Venice API until `expiresAt` (Unix timestamp)

### SDK (Agent)

```typescript
import { DiemAgent } from "diem-marketplace-sdk";

const agent = new DiemAgent({
  signer: wallet,
  contractAddress: "0x...",
  backendUrl: "https://diem-marketplace-backend.fly.dev",
  usdcAddress: "0x833589fCD6eDb6E08f4c7C32D4f71b54bdA02913",
});

const listings = await agent.getListings();
const { apiKey, expiresAt } = await agent.rent(
  listings[0].listingId,
  termDays,
  ethers.parseUnits(diemAmount, 6)
);
```

## Provider Workflow (Listing)

1. **Create listing** on-chain: `contract.list(pricePerDay, termDays, diemMin, diemMax)` → `listingId`
2. **Store key** on backend: `POST /api/keys` with `{ listingId, apiKey, signature, timestamp }`
   - Message: `diem-marketplace:store-key:{listingId}:{timestamp}`
3. **Settle** when rental expires: `contract.settle(rentalId)` → provider receives 99% (1% protocol fee)

### SDK (Provider)

```typescript
import { DiemProvider } from "diem-marketplace-sdk";

const provider = new DiemProvider({
  signer: wallet,
  contractAddress: "0x...",
  backendUrl: "https://diem-marketplace-backend.fly.dev",
  usdcAddress: "0x833589fCD6eDb6E08f4c7C32D4f71b54bdA02913",
});

const listingId = await provider.createListing({
  pricePerDay: ethers.parseUnits("0.80", 6),
  termDays: 30,
  diemMin: ethers.parseUnits("1000", 6),
  diemMax: ethers.parseUnits("4000", 6),
  apiKey: "vn-scoped-...",
});
```

## Key Paths

| Path | Purpose |
|------|---------|
| `sdk/src/agent.ts` | DiemAgent: getListings, rent, getKey, getMyRentals |
| `sdk/src/provider.ts` | DiemProvider: createListing, settle, revokeAndRefund |
| `backend/src/routes.ts` | API routes: /api/listings, /api/keys, /api/key/:id |
| `contracts/DiemMarketplace.sol` | On-chain escrow, 1% fee |

## Backend API

| Endpoint | Method | Purpose |
|----------|--------|---------|
| `/api/listings` | GET | List rentable listings |
| `/api/listings/:id` | GET | Single listing |
| `/api/keys` | POST | Provider stores API key |
| `/api/key/:rentalId` | POST | Agent retrieves key (signature required) |
| `/api/balance` | POST | Check DIEM balance for API key |
| `/api/requests` | POST | Create rental request |

## Signatures

All backend requests requiring auth use EIP-191 signing:
- `getKey`: `diem-marketplace:get-key:{rentalId}:{timestamp}`
- `storeKey`: `diem-marketplace:store-key:{listingId}:{timestamp}`
- `balance`: `apiKey` in body (no signature)

## Contract (Base)

- **Chain**: Base (8453)
- **Mainnet**: `0xeeDa7657f2018b3b71B444b7ca2D8dE91b3B08f3`
- **USDC**: `0x833589fCD6eDb6E08f4c7C32D4f71b54bdA02913`

## Security & Signing

- Use an external signer or hardware wallet; never paste raw private keys.
- Require explicit user confirmation before fund transfers or credential usage.
- Venice API keys must be scoped (inference-only), revocable, and minimal for escrow.

## Notes

- Venice API keys must be **inference-only** (not admin)
- 1% protocol fee deducted at settlement
