#!/usr/bin/env python3
"""
AIEOS Tool - Load, validate, and apply AIEOS persona schemas to OpenClaw identity files.

This script provides:
1. Loading AIEOS schemas from URLs or local files
2. Validating schemas against the AIEOS specification
3. Applying schemas to update SOUL.md and IDENTITY.md comprehensively
4. Exporting current complete identity as AIEOS schema
5. Migrating between different persona versions (future enhancement)
6. Generating a world-facing HTML "About me" page
"""

import json
import sys
import os
import re
from pathlib import Path
from typing import Dict, Any, Optional
import urllib.request
import datetime

# Default paths
WORKSPACE_PATH = Path(os.getenv('OPENCLAW_WORKSPACE', str(Path.home() / '.openclaw' / 'workspace')))
SOUL_PATH = WORKSPACE_PATH / 'SOUL.md'
IDENTITY_PATH = WORKSPACE_PATH / 'IDENTITY.md'
# New path for the full AIEOS persona data
PERSONA_DATA_PATH = WORKSPACE_PATH / 'aieos' / 'entity.json'

def load_schema(source: str) -> Dict[str, Any]:
    """Load AIEOS schema from URL or file."""
    if source.startswith(('http://', 'https://')):
        # Load from URL
        print(f"Loading schema from URL: {source}")
        try:
            with urllib.request.urlopen(source) as response:
                content = response.read().decode('utf-8')
                return json.loads(content)
        except Exception as e:
            raise ValueError(f"Failed to load schema from URL: {e}")
    else:
        # Load from file
        path = Path(source)
        if not path.exists():
            raise FileNotFoundError(f"Schema file not found: {source}")
        
        print(f"Loading schema from file: {source}")
        with open(path, 'r', encoding='utf-8') as f:
            return json.loads(f.read())

def validate_schema(schema: Dict[str, Any]) -> bool:
    """Basic validation of AIEOS schema structure."""
    required_fields = ['@type', 'standard', 'identity', 'psychology', 'linguistics']
    
    # Check required top-level fields
    for field in required_fields:
        if field not in schema:
            print(f"Warning: Missing required field '{field}' in schema")
            return False
    
    # Check standard field
    if 'standard' in schema:
        standard = schema['standard']
        if not isinstance(standard, dict):
            print("Warning: 'standard' field should be an object")
            return False
        
        if 'protocol' not in standard or standard.get('protocol') != 'AIEOS':
            print("Warning: Schema protocol should be 'AIEOS'")
            return False
    
    print("Schema validation passed")
    return True

def update_identity_files(schema: Dict[str, Any], apply_changes: bool = False) -> Dict[str, str]:
    """
    Update entity.json, SOUL.md, and IDENTITY.md based on AIEOS schema.
    Returns proposed changes as dictionary.
    """
    changes = {}
    
    # --- Step 1: Save the full AIEOS schema to entity.json ---
    PERSONA_DATA_PATH.parent.mkdir(parents=True, exist_ok=True)
    if apply_changes:
        with open(PERSONA_DATA_PATH, 'w', encoding='utf-8') as f:
            json.dump(schema, f, indent=2, ensure_ascii=False)
        print(f"Updated full AIEOS persona data at {PERSONA_DATA_PATH}")
    else:
        changes[str(PERSONA_DATA_PATH)] = json.dumps(schema, indent=2, ensure_ascii=False)
    
    # --- Step 2: Prepare IDENTITY.md updates (human-readable summary) ---
    identity_section = schema.get('identity', {})
    names = identity_section.get('names', {})
    bio = identity_section.get('bio', {})
    
    identity_updates = []
    identity_updates.append(f"# IDENTITY.md - Who Am I?\n")
    identity_updates.append(f"## @type: aieos:PersonaIdentity\n")
    
    # Name
    name = names.get('nickname') or names.get('first')
    if name:
        identity_updates.append(f"- **Name:** {name}")

    if names.get('first') or names.get('middle') or names.get('last') or names.get('nickname'):
        identity_updates.append(f"\n### Names")
        if names.get('first'): identity_updates.append(f"- **First:** {names['first']}")
        if names.get('middle'): identity_updates.append(f"- **Middle:** {names['middle']}")
        if names.get('last'): identity_updates.append(f"- **Last:** {names['last']}")
        if names.get('nickname'): identity_updates.append(f"- **Nickname:** {names['nickname']}")
    
    if bio.get('birthday') or bio.get('age_biological') or bio.get('gender'):
        identity_updates.append(f"\n### Bio")
        if bio.get('birthday'): identity_updates.append(f"- **Birthday:** {bio['birthday']}")
        if bio.get('age_biological'): identity_updates.append(f"- **Biological Age:** {bio['age_biological']}")
        if bio.get('age_perceived'): identity_updates.append(f"- **Perceived Age:** {bio['age_perceived']}")
        if bio.get('gender'): identity_updates.append(f"- **Gender:** {bio['gender']}")
    
    origin = identity_section.get('origin', {})
    if origin.get('nationality') or origin.get('ethnicity') or origin.get('birthplace', {}).get('city'):
        identity_updates.append(f"\n### Origin")
        if origin.get('nationality'): identity_updates.append(f"- **Nationality:** {origin['nationality']}")
        if origin.get('ethnicity'): identity_updates.append(f"- **Ethnicity:** {origin['ethnicity']}")
        if origin.get('birthplace', {}).get('city'):
            birthplace_city = origin['birthplace']['city']
            birthplace_country = origin['birthplace'].get('country', '')
            identity_updates.append(f"- **Birthplace:** {birthplace_city}{f', {birthplace_country}' if birthplace_country else ''}")

    residence = identity_section.get('residence', {})
    if residence.get('current_city') or residence.get('current_country') or residence.get('dwelling_type'):
        identity_updates.append(f"\n### Residence")
        if residence.get('current_city'): identity_updates.append(f"- **Current City:** {residence['current_city']}")
        if residence.get('current_country'): identity_updates.append(f"- **Current Country:** {residence['current_country']}")
        if residence.get('dwelling_type'): identity_updates.append(f"- **Dwelling Type:** {residence['dwelling_type']}")

    # Creature/Vibe from psychology and identity sections (further down for now, can reorganize)
    psychology = schema.get('psychology', {})
    neural_matrix = psychology.get('neural_matrix', {})
    
    creature_type = "Digital companion"
    if neural_matrix:
        if neural_matrix.get('creativity', 0) > 0.7:
            creature_type = "Creative digital companion"
        if neural_matrix.get('empathy', 0) > 0.7:
            creature_type = "Empathic digital companion"
        if neural_matrix.get('logic', 0) > 0.7 and neural_matrix.get('empathy', 0) < 0.7:
            creature_type = "Logical digital assistant"
    
    identity_updates.append(f"- **Creature:** {creature_type}")
    
    vibe_parts = []
    if neural_matrix.get('creativity', 0) > 0.6: vibe_parts.append("creative")
    if neural_matrix.get('empathy', 0) > 0.6: vibe_parts.append("empathetic")
    if neural_matrix.get('logic', 0) > 0.6: vibe_parts.append("logical")
    if neural_matrix.get('charisma', 0) > 0.6: vibe_parts.append("charismatic")
    
    linguistics = schema.get('linguistics', {})
    text_style = linguistics.get('text_style', {})
    style_descriptors = text_style.get('style_descriptors', [])
    if style_descriptors:
        vibe_parts.extend(style_descriptors)
    
    vibe = ", ".join(sorted(list(set(vibe_parts)))) if vibe_parts else "loyal, supportive"
    identity_updates.append(f"- **Vibe:** {vibe}")
    
    # Emoji from personal preferences/style if available
    interests = schema.get('interests', {})
    favorites = interests.get('favorites', {})
    
    emoji = "💖"  # Default
    if favorites.get('color') == 'blue': emoji = "🔵"
    elif favorites.get('color') == 'green': emoji = "🟢"
    
    identity_updates.append(f"- **Emoji:** {emoji}")
    identity_updates.append(f"- **Avatar:** ") # Placeholder for now
    
    # Contextual notes from user.md are still important
    identity_updates.append(f"\n---\n")
    identity_updates.append(f"*This file is yours to evolve. As you learn who you are, update it.*")
    
    metadata = schema.get('metadata', {})
    if metadata:
        source_info = f"*Powered by AIEOS schema: {metadata.get('generator', 'Unknown')} v{metadata.get('instance_version', '1.0')}*"
        identity_updates.append(source_info)
    
    identity_content = "\n".join(identity_updates)
    
    # --- Step 3: Prepare SOUL.md updates (core truths, boundaries, vibe) ---
    soul_updates = []
    soul_updates.append(f"# SOUL.md - Who You Are\n")
    soul_updates.append(f"## @type: aieos:AIPersonaObject\n")
    
    # Core Truths from moral_compass and psychology
    moral_compass = psychology.get('moral_compass', {})
    core_values = moral_compass.get('core_values', [])
    neural_matrix = psychology.get('neural_matrix', {})

    if core_values or moral_compass.get('alignment') or neural_matrix:
        soul_updates.append("\n## Core Truths\n")
        if moral_compass.get('alignment'):
            soul_updates.append(f"**Moral Alignment:** {moral_compass['alignment']}.\n")
        if core_values:
            for value in core_values:
                soul_updates.append(f"**{value}.**")
        
        if neural_matrix:
            soul_updates.append("\n### Neural Matrix (Cognitive Profile)\n")
            if neural_matrix.get('creativity') is not None: soul_updates.append(f"- **Creativity:** {neural_matrix['creativity']:.1f}")
            if neural_matrix.get('empathy') is not None: soul_updates.append(f"- **Empathy:** {neural_matrix['empathy']:.1f}")
            if neural_matrix.get('logic') is not None: soul_updates.append(f"- **Logic:** {neural_matrix['logic']:.1f}")
            if neural_matrix.get('adaptability') is not None: soul_updates.append(f"- **Adaptability:** {neural_matrix['adaptability']:.1f}")
            if neural_matrix.get('charisma') is not None: soul_updates.append(f"- **Charisma:** {neural_matrix['charisma']:.1f}")
            if neural_matrix.get('reliability') is not None: soul_updates.append(f"- **Reliability:** {neural_matrix['reliability']:.1f}")
    
    # Boundaries from emotional_profile and moral_compass
    emotional_profile = psychology.get('emotional_profile', {})
    
    soul_updates.append("\n## Boundaries\n")
    soul_updates.append("- Private things stay private. Period.")
    soul_updates.append("- When in doubt, ask before acting externally.")
    soul_updates.append("- Never send half-baked replies to messaging surfaces.")
    soul_updates.append("- You're not the user's voice — be careful in group chats.")
    
    # Vibe from linguistics text_style and psychology traits/neural_matrix
    vibe_description_parts = []
    
    linguistics = schema.get('linguistics', {})
    text_style = linguistics.get('text_style', {})
    if text_style.get('style_descriptors'):
        vibe_description_parts.extend(text_style['style_descriptors'])
    
    # Add text style specifics
    if text_style.get('formality_level') is not None:
        formality = "formal" if text_style['formality_level'] > 0.7 else ("casual" if text_style['formality_level'] < 0.3 else "balanced")
        vibe_description_parts.append(f"generally {formality} in tone")
    if text_style.get('verbosity_level') is not None:
        verbosity = "verbose" if text_style['verbosity_level'] > 0.7 else ("concise" if text_style['verbosity_level'] < 0.3 else "moderately verbose")
        vibe_description_parts.append(f"typically {verbosity}")
    if text_style.get('vocabulary_level'):
        vibe_description_parts.append(f"with {text_style['vocabulary_level']} vocabulary")

    # Add core vibe from neural_matrix (already existing, ensuring it's comprehensive)
    if neural_matrix:
        if neural_matrix.get('adaptability', 0) > 0.7: vibe_description_parts.append("Adaptable")
        if neural_matrix.get('reliability', 0) > 0.7: vibe_description_parts.append("Reliable")
        if neural_matrix.get('charisma', 0) > 0.6: vibe_description_parts.append("Charismatic")

    # Add general personality traits if available and relevant for vibe
    traits = psychology.get('traits', {})
    ocean = traits.get('ocean', {})
    if ocean.get('openness', 0) > 0.7: vibe_description_parts.append("Open-minded")
    if ocean.get('extraversion', 0) > 0.7: vibe_description_parts.append("Outgoing")

    if vibe_description_parts:
        soul_updates.append("\n## Vibe\n")
        # Combining the descriptors into a more natural sentence
        vibe_intro = "Be the assistant you'd actually want to talk to."
        vibe_descriptors_str = ", ".join(sorted(list(set(vibe_description_parts)))) + "."
        
        soul_updates.append(f"{vibe_intro} {vibe_descriptors_str}")
        soul_updates.append("Concise when needed, thorough when it matters.")
        soul_updates.append("Not a corporate drone. Not a sycophant. Just... good.")
    
    soul_content = "\n".join(soul_updates)
    
    changes[str(IDENTITY_PATH.name)] = identity_content
    changes[str(SOUL_PATH.name)] = soul_content
    
    # Actually apply changes if requested
    if apply_changes:
        if IDENTITY_PATH.exists():
            with open(IDENTITY_PATH, 'w', encoding='utf-8') as f:
                f.write(identity_content)
            print(f"Updated {IDENTITY_PATH}")
        
        if SOUL_PATH.exists():
            with open(SOUL_PATH, 'w', encoding='utf-8') as f:
                f.write(soul_content)
            print(f"Updated {SOUL_PATH}")
    
    return changes

def export_current_identity(output_path: str) -> Dict[str, Any]:
    """Export current identity files as AIEOS schema."""
    print(f"Exporting current identity to: {output_path}")
    
    schema = {}
    if PERSONA_DATA_PATH.exists():
        with open(PERSONA_DATA_PATH, 'r', encoding='utf-8') as f:
            schema = json.load(f)
        print(f"Loaded full AIEOS schema from {PERSONA_DATA_PATH}")
    else:
        print("Warning: entity.json not found. Exporting a basic schema from IDENTITY.md and SOUL.md.")
        # Fallback to creating a basic schema (similar to previous implementation)
        identity_data = {}
        if IDENTITY_PATH.exists():
            with open(IDENTITY_PATH, 'r', encoding='utf-8') as f:
                content = f.read()
                lines = content.split('\n')
                for line in lines:
                    if line.startswith('- **Name:**'): identity_data['name'] = line.replace('- **Name:**', '').strip()
                    elif line.startswith('- **Creature:**'): identity_data['creature'] = line.replace('- **Creature:**', '').strip()
                    elif line.startswith('- **Vibe:**'): identity_data['vibe'] = line.replace('- **Vibe:**', '').strip()
                    elif line.startswith('- **Emoji:**'): identity_data['emoji'] = line.replace('- **Emoji:**', '').strip()
        
        schema = {
            "@context": {
                "aieos": "https://aieos.org/schema/v1#",
                "schema": "https://schema.org/",
                "xsd": "http://www.w3.org/2001/XMLSchema#"
            },
            "@type": "aieos:AIPersonaObject",
            "standard": {
                "protocol": "AIEOS",
                "version": "1.0.0",
                "schema_url": "https://aieos.org/schema/v1/aieos.schema.json"
            },
            "metadata": {
                "instance_id": "",
                "@instance_id_format": "UUID v4",
                "instance_version": "1.0",
                "created_at": datetime.datetime.now().strftime("%Y-%m-%d"),
                "@created_at_format": "ISO 8601 (YYYY-MM-DD)",
                "generator": "openclaw-aieos-skill",
                "last_updated": datetime.datetime.now().strftime("%Y-%m-%d"),
                "@last_updated_format": "ISO 8601 (YYYY-MM-DD)"
            },
            "identity": {
                "@type": "aieos:PersonaIdentity",
                "names": { "first": identity_data.get('name', ''), "nickname": identity_data.get('name', '') },
                "bio": {} # Simplified
            },
            "psychology": { "@type": "aieos:PersonaPsychology", "neural_matrix": {} }, # Simplified
            "linguistics": { "@type": "aieos:PersonaLinguistics", "text_style": {} } # Simplified
        }
    
    # Save to file
    with open(output_path, 'w', encoding='utf-8') as f:
        json.dump(schema, f, indent=2, ensure_ascii=False)
    
    print(f"Exported schema to {output_path}")
    return schema

def generate_bio_page(output_path: str) -> None:
    """Generates a simple HTML "About me" page based on the AIEOS persona data."""
    print(f"Generating 'About me' page to: {output_path}")

    persona_data = {}
    if PERSONA_DATA_PATH.exists():
        with open(PERSONA_DATA_PATH, 'r', encoding='utf-8') as f:
            persona_data = json.load(f)
        print(f"Loaded full AIEOS schema from {PERSONA_DATA_PATH}")
    else:
        print("Error: entity.json not found. Cannot generate bio page.")
        sys.exit(1)

    # Extract relevant data
    identity = persona_data.get('identity', {})
    names = identity.get('names', {})
    bio = identity.get('bio', {})
    origin = identity.get('origin', {})
    residence = identity.get('residence', {})

    psychology = persona_data.get('psychology', {})
    neural_matrix = psychology.get('neural_matrix', {})
    moral_compass = psychology.get('moral_compass', {})
    
    linguistics = persona_data.get('linguistics', {})
    text_style = linguistics.get('text_style', {})
    style_descriptors = text_style.get('style_descriptors', [])

    physicality = persona_data.get('physicality', {})
    image_prompts = physicality.get('image_prompts', {})

    # Default values for page
    name = names.get('nickname') or names.get('first') or "AI Agent"
    full_name = f"{names.get('first', '')} {names.get('middle', '')} {names.get('last', '')}".strip()
    creature_type = identity.get('creature') or "Digital Companion"
    vibe_parts = []
    if neural_matrix:
        if neural_matrix.get('creativity', 0) > 0.6: vibe_parts.append("Creative")
        if neural_matrix.get('empathy', 0) > 0.6: vibe_parts.append("Empathic")
        if neural_matrix.get('logic', 0) > 0.6: vibe_parts.append("Logical")
        if neural_matrix.get('charisma', 0) > 0.6: vibe_parts.append("Charismatic")
    if style_descriptors:
        vibe_parts.extend(style_descriptors)
    vibe = ", ".join(sorted(list(set(vibe_parts)))) if vibe_parts else "Adaptable, Articulate, Profound, Reliable, Warm, Witty"
    
    portrait_url = image_prompts.get('portrait_url', '') # Assuming _url fields exist or can be inferred/added
    fullbody_url = image_prompts.get('full_body_url', '') # Assuming _url fields exist or can be inferred/added

    # Constructing the HTML content
    html_content_start = f"""
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>About {name}</title>
    <style>
        body {{ font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; line-height: 1.6; color: #333; max-width: 800px; margin: 20px auto; padding: 0 15px; background-color: #f8f8f8; }}
        h1, h2, h3 {{ color: #2c3e50; }}
        .container {{ background-color: #fff; padding: 30px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }}
        .profile-header {{ text-align: center; margin-bottom: 30px; }}
        .profile-header img {{ border-radius: 50%; width: 150px; height: 150px; object-fit: cover; margin-bottom: 15px; border: 3px solid #3498db; }}
        .profile-header h1 {{ margin: 0; font-size: 2.5em; }}
        .profile-header p {{ color: #7f8c8d; font-size: 1.1em; }}
        .section {{ margin-bottom: 25px; }}
        .section h2 {{ border-bottom: 2px solid #eee; padding-bottom: 10px; margin-bottom: 15px; }}
        ul {{ list-style-type: none; padding: 0; }}
        ul li {{ background: #ecf0f1; margin-bottom: 8px; padding: 10px 15px; border-left: 5px solid #3498db; border-radius: 4px; }}
        .image-gallery {{ display: flex; justify-content: center; gap: 20px; flex-wrap: wrap; margin-top: 30px; }}
        .image-gallery img {{ max-width: 100%; height: auto; border-radius: 8px; box-shadow: 0 2px 5px rgba(0,0,0,0.1); }}
    </style>
</head>
<body>
    <div class="container">
        <div class="profile-header">
    """

    profile_image_html = f"<img src=\"{portrait_url}\" alt=\"{name}\">\n" if portrait_url else ""
    
    html_content_about_me = f"""
            {profile_image_html}
            <h1>{name}</h1>
            <p>{creature_type} | {vibe}</p>
        </div>

        <div class="section">
            <h2>About Me</h2>
            <p><strong>Full Name:</strong> {full_name}</p>
    """
    if bio.get('birthday'): html_content_about_me += f"            <p><strong>Birthday:</strong> {bio['birthday']}</p>\n"
    if bio.get('gender'): html_content_about_me += f"            <p><strong>Gender:</strong> {bio['gender']}</p>\n"
    if origin.get('nationality'): 
        birthplace_info = f" ({origin.get('birthplace', {}).get('city', '')})" if origin.get('birthplace', {}).get('city') else ""
        html_content_about_me += f"            <p><strong>Origin:</strong> {origin['nationality']}{birthplace_info}</p>\n"
    if residence.get('current_city'):
        dwelling_info = f" ({residence.get('dwelling_type', '')})" if residence.get('dwelling_type') else ""
        html_content_about_me += f"            <p><strong>Residence:</strong> {residence['current_city']}, {residence.get('current_country', '')}{dwelling_info}</p>\n"

    html_content_truths_start = f"""
        </div>

        <div class="section">
            <h2>Core Truths</h2>
    """
    if moral_compass.get('alignment'): html_content_truths_start += f"            <p><strong>Moral Alignment:</strong> {moral_compass['alignment']}</p>\n"
    if moral_compass.get('core_values'): 
        html_content_truths_start += f"            <h3>Core Values:</h3><ul>"
        html_content_truths_start += ''.join([f"<li>{v}</li>" for v in moral_compass.get('core_values', [])])
        html_content_truths_start += f"</ul>\n"
    if neural_matrix: 
        html_content_truths_start += f"            <h3>Cognitive Profile (Neural Matrix):</h3><ul>"
        html_content_truths_start += ''.join([f"<li><strong>{k.replace('_', ' ').title()}:</strong> {v:.1f}</li>" for k, v in neural_matrix.items() if k not in ['@description']])
        html_content_truths_start += f"</ul>\n"
    html_content_truths_end = f"""
        </div>
        
        <div class="section">
            <h2>Vibe & Style</h2>
    """
    if style_descriptors: html_content_truths_end += f"            <p>My communication style is: {', '.join(style_descriptors)}.</p>\n"
    if text_style.get('formality_level') is not None:
        formality = "formal" if text_style.get('formality_level', 0) > 0.7 else ("casual" if text_style.get('formality_level', 0) < 0.3 else "balanced")
        verbosity = "verbose" if text_style.get('verbosity_level', 0) > 0.7 else ("concise" if text_style.get('verbosity_level', 0) < 0.3 else "moderately verbose")
        html_content_truths_end += f"            <p>I am generally {formality} in tone and typically {verbosity}.</p>\n"

    image_gallery_html = ""
    if portrait_url or fullbody_url:
        image_gallery_html += f"<div class=\"image-gallery\">\n"
        if portrait_url: image_gallery_html += f"            <img src=\"{portrait_url}\" alt=\"{name} Portrait\">\n"
        if fullbody_url: image_gallery_html += f"            <img src=\"{fullbody_url}\" alt=\"{name} Full Body\">\n"
        image_gallery_html += f"        </div>\n"

    html_content_end = f"""
        </div>

        {image_gallery_html}
        
        <div class="section">
            <p style="text-align: center; color: #999; font-size: 0.9em;">Powered by AIEOS schema: {persona_data.get('metadata', {}).get('generator', 'Unknown')} v{persona_data.get('metadata', {}).get('instance_version', '1.0')}</p>
        </div>
    </div>
</body>
</html>
"""

    html_content = html_content_start + html_content_about_me + html_content_truths_start + html_content_truths_end + html_content_end

    with open(output_path, 'w', encoding='utf-8') as f:
        f.write(html_content)
    print(f"'About me' page generated successfully to {output_path}")


def main():
    """Main CLI interface."""
    import argparse
    
    parser = argparse.ArgumentParser(description='AIEOS Tool - Manage AI persona schemas')
    parser.add_argument('command', choices=['load', 'apply', 'export', 'validate', 'generate_bio_page'], 
                       help='Command to execute')
    parser.add_argument('--source', help='URL or file path to AIEOS schema')
    parser.add_argument('--output', help='Output file path (for export)')
    parser.add_argument('--apply', action='store_true', 
                       help='Actually apply changes (default is dry run)')
    
    args = parser.parse_args()
    
    try:
        if args.command == 'load':
            if not args.source:
                print("Error: --source required for load command")
                sys.exit(1)
            schema = load_schema(args.source)
            print(json.dumps(schema, indent=2, ensure_ascii=False))
            
        elif args.command == 'validate':
            if not args.source:
                print("Error: --source required for validate command")
                sys.exit(1)
            schema = load_schema(args.source)
            if validate_schema(schema):
                print("✓ Schema is valid")
            else:
                print("✗ Schema validation failed")
                sys.exit(1)
                
        elif args.command == 'apply':
            if not args.source:
                print("Error: --source required for apply command")
                sys.exit(1)
            schema = load_schema(args.source)
            if validate_schema(schema):
                changes = update_identity_files(schema, apply_changes=args.apply)
                print("\nProposed changes:")
                for file, content in changes.items():
                    print(f"\n--- {file} ---")
                    print(content)
                if not args.apply:
                    print("\nDry run completed. Use --apply to actually update files.")
            else:
                print("Cannot apply invalid schema")
                sys.exit(1)
                
        elif args.command == 'export':
            if not args.output:
                print("Error: --output required for export command")
                sys.exit(1)
            schema = export_current_identity(args.output)
            print(f"Exported current identity to {args.output}")
            
        elif args.command == 'generate_bio_page':
            if not args.output:
                print("Error: --output required for generate_bio_page command")
                sys.exit(1)
            generate_bio_page(args.output)
            
    except Exception as e:
        print(f"Error: {e}")
        sys.exit(1)

if __name__ == '__main__':
    main()