#!/usr/bin/env python3
# Revenue Tracker - 每日收入汇总

import os
import json
from datetime import datetime
from pathlib import Path

# Get absolute path to script directory
SCRIPT_DIR = Path(__file__).parent.resolve()
LOG_FILE = SCRIPT_DIR / "logs" / "revenue.log"
REPORT_FILE = SCRIPT_DIR / "reports" / "daily_summary.md"

# Initialize
LOG_FILE.parent.mkdir(parents=True, exist_ok=True)
REPORT_FILE.parent.mkdir(parents=True, exist_ok=True)

def log_revenue(amount, source, description):
    """记录收入"""
    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    entry = {"timestamp": timestamp, "amount": amount, "source": source, "description": description}

    with open(LOG_FILE, "a", encoding="utf-8") as f:
        f.write(json.dumps(entry) + "\n")

    print(f"✓ Recorded: {amount} from {source}")

def get_today_revenue():
    """获取今日收入"""
    date_str = datetime.now().strftime("%Y-%m-%d")
    total = 0

    if LOG_FILE.exists():
        with open(LOG_FILE, "r", encoding="utf-8") as f:
            for line in f:
                try:
                    entry = json.loads(line)
                    if entry["timestamp"].startswith(date_str):
                        total += float(entry["amount"])
                except json.JSONDecodeError:
                    pass

    return total

def get_total_revenue():
    """获取总收入"""
    total = 0

    if LOG_FILE.exists():
        with open(LOG_FILE, "r", encoding="utf-8") as f:
            for line in f:
                try:
                    entry = json.loads(line)
                    total += float(entry["amount"])
                except json.JSONDecodeError:
                    pass

    return total

def generate_report():
    """生成每日报告"""
    date_str = datetime.now().strftime("%Y-%m-%d")
    today_rev = get_today_revenue()
    total_rev = get_total_revenue()

    today_count = 0
    entries = []
    if LOG_FILE.exists():
        with open(LOG_FILE, "r", encoding="utf-8") as f:
            for line in f:
                try:
                    entry = json.loads(line)
                    if date_str in entry["timestamp"]:
                        today_count += 1
                        entries.append(entry)
                except json.JSONDecodeError:
                    pass

    with open(REPORT_FILE, "w", encoding="utf-8") as f:
        f.write(f"# 每日收入汇总\n\n")
        f.write(f"**日期:** {date_str}\n")
        f.write(f"**总收入:** \${today_rev:.2f}\n")
        f.write(f"**总交易数:** {today_count}\n")
        f.write(f"**累计总收入:** \${total_rev:.2f}\n\n")
        f.write("---\n\n")
        f.write("## 今日交易\n\n")

        for entry in entries:
            f.write(f"• {entry['timestamp']} | {entry['amount']} | {entry['source']} | {entry['description']}\n")

        f.write("\n---\n*Generated by Revenue Tracker*")

    print(f"Report saved to: {REPORT_FILE}")

# Main
if __name__ == "__main__":
    import sys

    if len(sys.argv) > 1:
        if sys.argv[1] == "log":
            if len(sys.argv) >= 4:
                log_revenue(sys.argv[2], sys.argv[3], sys.argv[4] if len(sys.argv) > 4 else "")
            else:
                print("Usage: python3 main.py log <金额> <来源> [描述]")
        elif sys.argv[1] == "report":
            generate_report()
        elif sys.argv[1] == "today":
            print(f"今日收入: \${get_today_revenue():.2f}")
        elif sys.argv[1] == "total":
            print(f"总收入: \${get_total_revenue():.2f}")
        else:
            print("Usage: python3 main.py {log|report|today|total}")
    else:
        print("Usage: python3 main.py {log|report|today|total}")
