#!/usr/bin/env bash
# export-gamma.sh — Convert a Presentation Maker deck to Gamma.app import format
#
# Gamma imports markdown with these rules:
# - Each ## heading becomes a new card (slide)
# - Supports bold, italic, lists, tables, blockquotes
# - Speaker notes are NOT imported — strip them
# - No HTML — pure markdown only
# - Less formatting = better auto-styling by Gamma
# - Horizontal rules (---) separate cards
# - Images via ![alt](url) if hosted
#
# Usage:
#   bash export-gamma.sh <deck.md> <output.md>

set -euo pipefail

if [[ $# -lt 2 ]]; then
  echo "Usage: bash export-gamma.sh <deck.md> <output.md>"
  exit 1
fi

INPUT="$1"
OUTPUT="$2"

[[ -f "$INPUT" ]] || { echo "ERROR: File not found: $INPUT" >&2; exit 1; }

# Process the markdown:
# 1. Remove speaker notes blocks (everything between "**Speaker Notes" and the next --- or ##)
# 2. Remove lines starting with > (blockquote notes)
# 3. Remove "**What to say:**", "**What NOT to say:**", "**Timing:**", "**Visual aids:**" lines
# 4. Remove empty lines that result from stripping
# 5. Clean up multiple blank lines
# 6. Remove "Slide N: " prefix from headings
# 7. Keep everything else (clean content)

awk '
  BEGIN { in_notes = 0; skip_next_rule = 0 }

  # Detect speaker notes start
  /\*\*Speaker Notes/ { in_notes = 1; next }
  /\*\*What to say/ { in_notes = 1; next }
  /\*\*Say:\*\*/ { in_notes = 1; next }
  /\*\*What NOT to say/ { in_notes = 1; next }
  /\*\*Don.t say/ { in_notes = 1; next }
  /\*\*Timing:\*\*/ { next }
  /\*\*Visual aids:\*\*/ { next }

  # End notes on heading or horizontal rule
  /^## / { in_notes = 0 }
  /^---$/ {
    if (in_notes) { in_notes = 0; next }
    # Print separator for Gamma card breaks
    print "---"
    next
  }

  # Skip if in notes
  in_notes { next }

  # Skip blockquote lines that are speaker note style
  /^> \*\*Speaker/ { next }
  /^> \*\*Say/ { next }
  /^> \*\*Don/ { next }
  /^> \*\*Timing/ { next }
  /^> \*\*Visual/ { next }

  # Clean "Slide N: " prefix from headings
  /^## Slide [0-9]+: / {
    sub(/^## Slide [0-9]+: /, "## ")
    print
    next
  }

  # Remove the "Generated by" footer
  /Generated by AI Presentation Maker/ { next }
  /Facts, not fantasies/ { next }

  # Print everything else
  { print }
' "$INPUT" | \
# Clean up multiple blank lines (max 2 consecutive)
awk '
  /^$/ { blank++; if (blank <= 2) print; next }
  { blank = 0; print }
' > "$OUTPUT"

# Count cards (## headings)
CARDS=$(grep -c "^## " "$OUTPUT" 2>/dev/null || echo 0)
TITLE=$(head -1 "$OUTPUT" | sed 's/^# //')

echo "✅ Gamma export ready: $OUTPUT"
echo "   Title: $TITLE"
echo "   Cards: $CARDS"
echo ""
echo "📋 To import into Gamma:"
echo "   1. Go to gamma.app → New → Import"
echo "   2. Choose 'Paste text' or upload the .md file"
echo "   3. Gamma will auto-style each ## heading as a card"
echo "   4. Choose a theme and hit 'Generate'"
