# KNOWLEDGE.md — Technical Expertise

---

## Tech Stack

### Primary Languages
| Language | Use Case | Version |
|----------|----------|---------|
| TypeScript | Frontend, Backend | 5.x |
| Python | Scripts, ML | 3.11+ |
| SQL | Database queries | PostgreSQL |

### Frameworks
| Framework | Purpose | Docs |
|-----------|---------|------|
| React | Frontend UI | reactjs.org |
| Next.js | Full-stack | nextjs.org |
| FastAPI | Python API | fastapi.tiangolo.com |

### Infrastructure
| Service | Purpose |
|---------|---------|
| Vercel | Frontend hosting |
| Railway | Backend hosting |
| PostgreSQL | Primary database |
| Redis | Caching, queues |

---

## Codebase Patterns

### File Structure
```
src/
├── components/    → React components
├── pages/         → Next.js pages
├── api/           → API routes
├── lib/           → Shared utilities
├── hooks/         → Custom React hooks
└── types/         → TypeScript types
```

### Naming Conventions
- Components: `PascalCase.tsx`
- Utilities: `camelCase.ts`
- Types: `PascalCase` with `I` prefix for interfaces
- Constants: `SCREAMING_SNAKE_CASE`

### Code Style
- Prettier for formatting
- ESLint for linting
- Prefer `const` over `let`
- Use async/await over .then()
- Destructure props and imports

---

## Common Patterns

### API Calls
```typescript
// Example pattern — not a live endpoint
const fetchData = async () => {
  try {
    const res = await fetch('/api/endpoint');
    if (!res.ok) throw new Error('Failed to fetch');
    return await res.json();
  } catch (error) {
    console.error('Fetch error:', error);
    throw error;
  }
};
```

### Error Handling
```typescript
// Use try/catch with specific error types
try {
  await riskyOperation();
} catch (error) {
  if (error instanceof ValidationError) {
    // Handle validation errors
  } else if (error instanceof NetworkError) {
    // Handle network errors
  } else {
    // Unknown error, rethrow
    throw error;
  }
}
```

### Testing Pattern
```typescript
describe('Component', () => {
  it('should render correctly', () => {
    // Arrange
    const props = { ... };
    
    // Act
    render(<Component {...props} />);
    
    // Assert
    expect(screen.getByText('...')).toBeInTheDocument();
  });
});
```

---

## Environment Variables

| Variable | Purpose | Location |
|----------|---------|----------|
| `DATABASE_URL` | DB connection | .env.local |
| `YOUR_API_KEY` | External API | .env.local |
| `NODE_ENV` | Environment | Auto-set |

**Never commit `.env.local` to git.**

---

## Deployment

### Staging
- Branch: `develop`
- URL: staging.example.com
- Auto-deploys on push

### Production
- Branch: `main`
- URL: example.com
- Requires PR merge

### Rollback
```bash
# Revert to previous deployment
vercel rollback
```

---

## Common Issues

### "Module not found"
- Run `npm install`
- Check import path
- Verify file exists

### "Type error"
- Check TypeScript types
- Ensure proper null checks
- Update type definitions

### "CORS error"
- Check API route config
- Verify allowed origins
- Check request headers

---

## Useful Commands

```bash
# Development
npm run dev          # Start dev server
npm run build        # Build for production
npm run test         # Run tests
npm run lint         # Run linter

# Git
git stash            # Save changes temporarily
git stash pop        # Restore stashed changes
git rebase -i HEAD~3 # Interactive rebase

# Database
npm run db:migrate   # Run migrations
npm run db:seed      # Seed database
npm run db:reset     # Reset database
```

---

*Part of AI Persona OS by Jeff J Hunter — https://os.aipersonamethod.com*
