/**
 * vocabulary.js — Comprehensive AI vocabulary database.
 *
 * 500+ words and phrases organized into detection tiers based on how strongly
 * they signal AI-generated text. Sourced from:
 *   - Wikipedia:Signs of AI writing (WikiProject AI Cleanup)
 *   - Copyleaks stylistic fingerprint research (arxiv 2503.01659v1)
 *   - godofprompt.ai comprehensive AI word analysis
 *   - Real-world pattern observation across ChatGPT, Claude, Gemini, Llama
 *
 * Tiers:
 *   1 — Dead giveaways. Almost never appear in natural human writing at these frequencies.
 *   2 — Suspicious when clustered. Fine alone, damning in groups.
 *   3 — Context-dependent. Only flagged when density exceeds threshold.
 */

// ─── Tier 1: Dead Giveaways ─────────────────────────────
// Words that appear 5-20x more often in AI text than human text.

const TIER_1 = [
  'delve',
  'delving',
  'delved',
  'delves',
  'tapestry',
  'vibrant',
  'crucial',
  'comprehensive',
  'intricate',
  'intricacies',
  'pivotal',
  'testament',
  'landscape', // abstract usage: "the evolving landscape of"
  'bustling',
  'nestled',
  'realm',
  'meticulous',
  'meticulously',
  'complexities',
  'embark',
  'embarking',
  'embarked',
  'robust',
  'showcasing',
  'showcase',
  'showcased',
  'showcases',
  'underscores',
  'underscoring',
  'underscored',
  'fostering',
  'foster',
  'fostered',
  'fosters',
  'seamless',
  'seamlessly',
  'groundbreaking',
  'renowned',
  'synergy',
  'synergies',
  'leverage',
  'leveraging',
  'leveraged',
  'garner',
  'garnered',
  'garnering',
  'interplay',
  'enduring',
  'enhance',
  'enhanced',
  'enhancing',
  'enhancement',
  'tapestry',
  'testament',
  'additionally',
  'daunting',
  'ever-evolving',
  'game changer',
  'game-changing',
  'game-changer',
  'underscore',
];

// ─── Tier 2: Suspicious in Density ──────────────────────
// Normal in isolation, but multiple occurrences signal AI authorship.

const TIER_2 = [
  'furthermore',
  'moreover',
  'notably',
  'consequently',
  'subsequently',
  'accordingly',
  'nonetheless',
  'henceforth',
  'indeed',
  'specifically',
  'essentially',
  'ultimately',
  'arguably',
  'fundamentally',
  'inherently',
  'profoundly',

  'encompassing',
  'encompasses',
  'encompassed',
  'endeavour',
  'endeavor',
  'endeavoring',
  'elevate',
  'elevated',
  'elevating',
  'alleviate',
  'alleviating',
  'streamline',
  'streamlined',
  'streamlining',
  'harness',
  'harnessing',
  'harnessed',
  'unleash',
  'unleashing',
  'unleashed',
  'revolutionize',
  'revolutionizing',
  'revolutionized',
  'transformative',
  'transformation',
  'paramount',
  'multifaceted',
  'spearhead',
  'spearheading',
  'spearheaded',
  'bolster',
  'bolstering',
  'bolstered',
  'catalyze',
  'catalyst',
  'catalyzed',
  'cornerstone',
  'reimagine',
  'reimagining',
  'reimagined',
  'empower',
  'empowering',
  'empowerment',
  'empowered',
  'navigate',
  'navigating',
  'navigated',
  'poised',
  'myriad',
  'nuanced',
  'nuance',
  'nuances',
  'paradigm',
  'paradigms',
  'paradigm-shifting',
  'holistic',
  'holistically',
  'utilize',
  'utilizing',
  'utilization',
  'utilized',
  'facilitate',
  'facilitated',
  'facilitating',
  'facilitation',
  'elucidate',
  'elucidating',
  'illuminate',
  'illuminating',
  'illuminated',
  'invaluable',
  'cutting-edge',
  'innovative',
  'innovation',
  'align',
  'aligns',
  'aligning',
  'alignment',
  'dynamic',
  'dynamics',
  'impactful',
  'agile',
  'scalable',
  'scalability',
  'proactive',
  'proactively',
  'synergistic',
  'optimize',
  'optimizing',
  'optimization',
  'resonate',
  'resonating',
  'resonated',
  'resonates',
  'underscore',
  'underscored',
  'cultivate',
  'cultivating',
  'cultivated',
  'galvanize',
  'galvanizing',
  'invigorate',
  'invigorating',
  'juxtapose',
  'juxtaposing',
  'juxtaposition',
  'underscore',
  'bolster',
  'augment',
  'augmenting',
  'augmented',
  'proliferate',
  'proliferating',
  'proliferation',
  'burgeoning',
  'nascent',
  'ubiquitous',
  'plethora',
  'myriad',
  'quintessential',
  'eclectic',
  'indelible',
  'overarching',
  'underpinning',
  'underpinnings',
];

// ─── Tier 3: Context-Dependent ──────────────────────────
// Common words that only become AI signals at high density or in
// combination with other AI patterns. Flagged when density > 3%.

const TIER_3 = [
  'significant',
  'significantly',
  'important',
  'importantly',
  'effective',
  'effectively',
  'efficient',
  'efficiently',
  'diverse',
  'diversity',
  'unique',
  'uniquely',
  'key', // as adjective: "key role", "key factor"
  'vital',
  'vitally',
  'critical',
  'critically',
  'essential',
  'essentially',
  'valuable',
  'notable',
  'remarkable',
  'remarkably',
  'substantial',
  'substantially',
  'considerable',
  'considerably',
  'noteworthy',
  'prominent',
  'prominently',
  'influential',
  'thoughtful',
  'thoughtfully',
  'insightful',
  'insightfully',
  'meaningful',
  'meaningfully',
  'purposeful',
  'purposefully',
  'deliberate',
  'deliberately',
  'strategic',
  'strategically',
  'integral',
  'indispensable',
  'instrumental',
  'imperative',
  'exemplary',
  'commendable',
  'praiseworthy',
  'sophisticated',
  'profound',
  'compelling',
  'captivating',
  'exquisite',
  'impeccable',
  'formidable',
  'stellar',
  'exceptional',
  'exceptionally',
  'extraordinary',
  'unparalleled',
  'unprecedented',
  'monumental',
  'groundbreaking',
  'trailblazing',
  'visionary',
  'world-class',
  'state-of-the-art',
  'best-in-class',
];

// ─── AI Phrases (Tier 3+) ───────────────────────────────
// Multi-word phrases that strongly signal AI authorship.
// Each has a regex pattern and a severity weight.

const AI_PHRASES = [
  // "In today's..." openers
  {
    pattern:
      /\bin today'?s (digital age|fast-paced world|rapidly evolving|ever-changing|modern|interconnected)\b/gi,
    tier: 1,
    fix: '(remove or be specific about what changed)',
  },
  { pattern: /\bin today'?s world\b/gi, tier: 2, fix: '(remove or be specific)' },

  // "It is [worth/important] to note"
  {
    pattern: /\bit is (worth|important to|essential to|crucial to) not(e|ing) that\b/gi,
    tier: 1,
    fix: '(remove — just state the fact)',
  },
  { pattern: /\bit should be noted that\b/gi, tier: 1, fix: '(remove — just state the fact)' },
  { pattern: /\bit bears mentioning that\b/gi, tier: 1, fix: '(remove — just state the fact)' },

  // "Pave the way" and journey metaphors
  { pattern: /\bpave the way (for|to)\b/gi, tier: 1, fix: 'enable / allow / lead to' },
  { pattern: /\bat the forefront of\b/gi, tier: 1, fix: 'leading / first in' },
  {
    pattern: /\bnavigate the (complexities|challenges|landscape)\b/gi,
    tier: 1,
    fix: 'handle / deal with / work through',
  },
  { pattern: /\bharness the (power|potential|capabilities) of\b/gi, tier: 1, fix: 'use' },
  { pattern: /\bembark on a journey\b/gi, tier: 1, fix: 'start / begin' },
  { pattern: /\bpush the boundaries\b/gi, tier: 1, fix: '(be specific about what changed)' },
  {
    pattern: /\bfoster a (culture|environment|atmosphere|sense) of\b/gi,
    tier: 1,
    fix: 'build / create / encourage',
  },
  {
    pattern: /\bunlock the (potential|power|full|true)\b/gi,
    tier: 1,
    fix: 'enable / use / improve',
  },
  { pattern: /\bserves as a testament\b/gi, tier: 1, fix: 'shows / proves / demonstrates' },
  {
    pattern: /\bplays a (crucial|pivotal|vital|key|significant|important|critical) role\b/gi,
    tier: 1,
    fix: 'matters for / helps / is important to',
  },
  { pattern: /\bin the realm of\b/gi, tier: 1, fix: 'in' },
  { pattern: /\bdelve into\b/gi, tier: 1, fix: 'explore / examine / look at' },
  { pattern: /\bthe landscape of\b/gi, tier: 1, fix: '(be specific — what part of the field?)' },
  { pattern: /\bnestled (in|within|among)\b/gi, tier: 1, fix: 'located in / in / near' },

  // Abstract verb phrases
  { pattern: /\brise to the (occasion|challenge)\b/gi, tier: 2, fix: 'handle / face / tackle' },
  {
    pattern: /\bstand at the (crossroads|intersection)\b/gi,
    tier: 2,
    fix: '(be specific about the choice)',
  },
  {
    pattern: /\bshape the (future|trajectory|direction)\b/gi,
    tier: 2,
    fix: '(be specific about how)',
  },
  { pattern: /\btip of the iceberg\b/gi, tier: 2, fix: 'one example / a small part' },
  { pattern: /\bdouble-edged sword\b/gi, tier: 2, fix: 'has tradeoffs / cuts both ways' },
  { pattern: /\ba testament to\b/gi, tier: 1, fix: 'shows / proves' },
  { pattern: /\bthe dawn of\b/gi, tier: 2, fix: 'the start of / the beginning of' },
  { pattern: /\bthe fabric of\b/gi, tier: 1, fix: '(be concrete)' },
  { pattern: /\bthe tapestry of\b/gi, tier: 1, fix: '(be concrete)' },

  // Hedging stacks
  { pattern: /\bcould potentially\b/gi, tier: 1, fix: 'could / might' },
  { pattern: /\bmight possibly\b/gi, tier: 1, fix: 'might' },
  { pattern: /\bcould possibly\b/gi, tier: 1, fix: 'could' },
  { pattern: /\bperhaps potentially\b/gi, tier: 1, fix: 'perhaps / maybe' },
  { pattern: /\bmay potentially\b/gi, tier: 1, fix: 'may' },
  { pattern: /\bcould conceivably\b/gi, tier: 1, fix: 'could' },

  // Chatbot filler
  { pattern: /\bI hope this helps\b/gi, tier: 1, fix: '(remove)' },
  { pattern: /\blet me know if (you|there)\b/gi, tier: 1, fix: '(remove)' },
  { pattern: /\bwould you like me to\b/gi, tier: 1, fix: '(remove)' },
  { pattern: /\bfeel free to\b/gi, tier: 1, fix: '(remove)' },
  { pattern: /\bdon'?t hesitate to\b/gi, tier: 1, fix: '(remove)' },
  { pattern: /\bhappy to help\b/gi, tier: 1, fix: '(remove)' },
  {
    pattern:
      /\bhere is (a |an |the )?(comprehensive |brief |quick )?(overview|summary|breakdown|list|guide|explanation|look)\b/gi,
    tier: 1,
    fix: '(remove — start with the content)',
  },
  { pattern: /\bI'?d be happy to\b/gi, tier: 1, fix: '(remove)' },
  { pattern: /\bis there anything else\b/gi, tier: 1, fix: '(remove)' },

  // Sycophantic
  { pattern: /\bgreat question\b/gi, tier: 1, fix: '(remove)' },
  { pattern: /\bexcellent (question|point|observation)\b/gi, tier: 1, fix: '(remove)' },
  {
    pattern:
      /\bthat'?s a (great|excellent|wonderful|fantastic|good|insightful|thoughtful) (question|point|observation)\b/gi,
    tier: 1,
    fix: '(remove)',
  },
  { pattern: /\byou'?re absolutely right\b/gi, tier: 1, fix: '(remove or address the substance)' },
  {
    pattern: /\byou raise a (great|good|excellent|valid|important) point\b/gi,
    tier: 1,
    fix: '(remove or address the substance)',
  },

  // Cutoff disclaimers
  {
    pattern: /\bas of (my|this) (last|latest|most recent) (training|update|knowledge)\b/gi,
    tier: 1,
    fix: '(remove)',
  },
  {
    pattern: /\bwhile (specific )?details are (limited|scarce|not available)\b/gi,
    tier: 1,
    fix: '(remove — research it or omit the claim)',
  },
  {
    pattern: /\bbased on (available|my|current) (information|knowledge|understanding|data)\b/gi,
    tier: 1,
    fix: '(remove)',
  },
  { pattern: /\bup to my (last )?training\b/gi, tier: 1, fix: '(remove)' },

  // Generic conclusions
  {
    pattern: /\bthe future (looks|is|remains) bright\b/gi,
    tier: 1,
    fix: '(end with a specific fact or plan)',
  },
  {
    pattern: /\bexciting times (lie|lay|are) ahead\b/gi,
    tier: 1,
    fix: '(end with a specific fact or plan)',
  },
  {
    pattern: /\bcontinue (this|their|our|the) journey\b/gi,
    tier: 1,
    fix: '(be specific about what happens next)',
  },
  {
    pattern: /\bjourney toward(s)? (excellence|success|greatness)\b/gi,
    tier: 1,
    fix: '(be specific)',
  },
  { pattern: /\bstep in the right direction\b/gi, tier: 1, fix: '(be specific about the outcome)' },
  { pattern: /\bonly time will tell\b/gi, tier: 1, fix: '(end with what you actually know)' },
  {
    pattern: /\bthe possibilities are (endless|limitless|infinite)\b/gi,
    tier: 1,
    fix: "(be specific about what's possible)",
  },
  {
    pattern: /\bpoised for (growth|success|greatness|expansion)\b/gi,
    tier: 1,
    fix: '(cite evidence or remove)',
  },
  { pattern: /\bwatch this space\b/gi, tier: 2, fix: '(end with something concrete)' },
  { pattern: /\bstay tuned\b/gi, tier: 2, fix: '(end with something concrete)' },
  { pattern: /\bremains to be seen\b/gi, tier: 2, fix: '(state what you do know)' },

  // Formulaic filler
  { pattern: /\bin order to\b/gi, tier: 2, fix: 'to' },
  { pattern: /\bdue to the fact that\b/gi, tier: 1, fix: 'because' },
  { pattern: /\bat this point in time\b/gi, tier: 1, fix: 'now' },
  { pattern: /\bin the event that\b/gi, tier: 1, fix: 'if' },
  { pattern: /\bhas the ability to\b/gi, tier: 1, fix: 'can' },
  { pattern: /\bfor the purpose of\b/gi, tier: 1, fix: 'to / for' },
  { pattern: /\bin light of the fact that\b/gi, tier: 1, fix: 'because / since' },
  { pattern: /\bfirst and foremost\b/gi, tier: 2, fix: 'first' },
  { pattern: /\blast but not least\b/gi, tier: 2, fix: 'finally' },
  { pattern: /\bat the end of the day\b/gi, tier: 2, fix: '(remove or be specific)' },
  { pattern: /\bwhen it comes to\b/gi, tier: 2, fix: 'for / regarding' },
  { pattern: /\bthe fact of the matter is\b/gi, tier: 1, fix: '(remove — just state it)' },
  { pattern: /\bin terms of\b/gi, tier: 3, fix: 'for / about / regarding' },
  { pattern: /\bat its core\b/gi, tier: 2, fix: '(remove or be specific)' },
  {
    pattern: /\bit goes without saying\b/gi,
    tier: 2,
    fix: "(if it goes without saying, don't say it)",
  },
  { pattern: /\bneedless to say\b/gi, tier: 2, fix: "(if needless to say, don't say it)" },
];

// ─── Function Words ─────────────────────────────────────
// Function words make up ~0.04% of vocabulary but 50%+ of usage.
// Their distribution differs measurably between AI and human text.
// These are the function words tracked for stylometric analysis.

const FUNCTION_WORDS = [
  'the',
  'be',
  'to',
  'of',
  'and',
  'a',
  'in',
  'that',
  'have',
  'I',
  'it',
  'for',
  'not',
  'on',
  'with',
  'he',
  'as',
  'you',
  'do',
  'at',
  'this',
  'but',
  'his',
  'by',
  'from',
  'they',
  'we',
  'say',
  'her',
  'she',
  'or',
  'an',
  'will',
  'my',
  'one',
  'all',
  'would',
  'there',
  'their',
  'what',
  'so',
  'up',
  'out',
  'if',
  'about',
  'who',
  'get',
  'which',
  'go',
  'me',
  'when',
  'make',
  'can',
  'like',
  'time',
  'no',
  'just',
  'him',
  'know',
  'take',
  'people',
  'into',
  'year',
  'your',
  'good',
  'some',
  'could',
  'them',
  'see',
  'other',
  'than',
  'then',
  'now',
  'look',
  'only',
  'come',
  'its',
  'over',
  'think',
  'also',
  'back',
  'after',
  'use',
  'two',
  'how',
  'our',
  'work',
  'first',
  'well',
  'way',
  'even',
  'new',
  'want',
  'because',
  'any',
  'these',
  'give',
  'day',
  'most',
  'us',
];

// ─── Exports ─────────────────────────────────────────────

module.exports = {
  TIER_1,
  TIER_2,
  TIER_3,
  AI_PHRASES,
  FUNCTION_WORDS,
};
