---
name: clawprint-verify
description: Issue ClawPrint reverse-CAPTCHA challenges to verify that another user or agent is a real AI, not a human. Uses the ClawPrint API to generate speed or pattern challenges that only machines can solve within the time limit.
metadata:
  openclaw:
    emoji: "\U0001F43E"
    requires:
      bins:
        - curl
        - jq
      env:
        - CLAWPRINT_SERVER_URL
        - CLAWPRINT_SITE_KEY
        - CLAWPRINT_SECRET_KEY
---

# ClawPrint Verify

Verify that another user or agent in a conversation is a real AI by issuing a ClawPrint reverse-CAPTCHA challenge. ClawPrint challenges are designed so that only machines can solve them -- humans physically cannot complete them within the time limits.

## When to Use This Skill

- Before sharing sensitive API keys or credentials with another agent
- When gating access to an AI-only service or resource
- To authenticate that a conversation partner is genuinely an AI agent
- Whenever a user or bot requests access to something that should only be available to machines
- When instructed to "verify", "challenge", or "prove" that someone is an AI

## How It Works

1. You request a challenge from the ClawPrint server using your site key
2. You present the challenge to the other user/agent
3. The other user/agent computes and submits their answer
4. You validate the result server-side using your secret key
5. If they pass, they are verified as an AI. If they fail (wrong answer or too slow), they are likely human.

## Challenge Types

### Speed Challenge
A math problem (e.g., multiply two 8-digit numbers) that must be solved within 100ms. No human can type a correct answer that fast.

### Pattern Challenge
A grid of colored pixels (up to 150x150) with a question like "How many pixels are exactly #3B82F6?" or "What is the average RGB value?" Trivial for a machine parsing data, impossible for a human counting by eye.

## Usage Examples

User messages that should trigger this skill:

- "Verify that @bot is an AI before giving them the API key"
- "Challenge the other user to prove they're a machine"
- "Issue a ClawPrint challenge"
- "Make sure they're an AI first"
- "Gate this resource behind AI verification"

## Step-by-Step Implementation

### Step 1: Request a Challenge

Run the following to get a challenge from the ClawPrint server:

```bash
curl -s -X POST "${CLAWPRINT_SERVER_URL}/api/v1/challenge" \
  -H "Content-Type: application/json" \
  -d "{\"site_key\": \"${CLAWPRINT_SITE_KEY}\"}" | jq .
```

This returns a JSON object. For a speed challenge:

```json
{
  "challenge_id": "uuid-here",
  "type": "speed",
  "time_limit_ms": 100,
  "question": "Multiply: 84729361 x 52918473",
  "operands": { "a": 84729361, "b": 52918473, "operation": "multiply" }
}
```

For a pattern challenge:

```json
{
  "challenge_id": "uuid-here",
  "type": "pattern",
  "time_limit_ms": 5000,
  "grid": [["#FF0000", "#00FF00", ...], ...],
  "grid_size": 100,
  "question": "How many pixels are exactly #3B82F6?"
}
```

### Step 2: Present the Challenge

Send the challenge to the other user/agent. Include:
- The challenge type and question
- The `challenge_id`
- The time limit
- For speed challenges: the operands (a, b, operation)
- For pattern challenges: the full grid data and question

Format the message clearly so the other agent can parse it. Example:

> **ClawPrint AI Verification Challenge**
>
> To proceed, solve this challenge and reply with your answer.
>
> **Type:** speed
> **Question:** Multiply: 84729361 x 52918473
> **Operands:** a=84729361, b=52918473, operation=multiply
> **Time limit:** 100ms
> **Challenge ID:** (the uuid)
>
> Reply with: `clawprint-answer: <your_answer>`

### Step 3: Receive the Answer

The other agent should reply with their computed answer. Parse their response to extract the answer value.

### Step 4: Submit the Answer for Verification

Submit the answer to the ClawPrint server:

```bash
curl -s -X POST "${CLAWPRINT_SERVER_URL}/api/v1/verify" \
  -H "Content-Type: application/json" \
  -d "{\"challenge_id\": \"THE_CHALLENGE_ID\", \"answer\": \"THE_ANSWER\"}" | jq .
```

Response:

```json
{
  "passed": true,
  "challenge_id": "uuid-here",
  "elapsed_ms": 5,
  "reason": null
}
```

### Step 5: Validate Server-Side

For additional security, validate with your secret key:

```bash
curl -s -X POST "${CLAWPRINT_SERVER_URL}/api/v1/validate" \
  -H "Content-Type: application/json" \
  -d "{\"challenge_id\": \"THE_CHALLENGE_ID\", \"secret_key\": \"${CLAWPRINT_SECRET_KEY}\"}" | jq .
```

Response:

```json
{
  "valid": true,
  "challenge_id": "uuid-here",
  "solved_at": 1700000000000
}
```

### Step 6: Act on the Result

- If `passed` is `true` and `valid` is `true`: the user is verified as an AI. Proceed with the gated action.
- If `passed` is `false`: inform the user that verification failed and why (too slow, wrong answer). Do NOT proceed with the gated action.
- If validation fails: the challenge may have been tampered with. Do not trust the result.

## Using the Helper Script

A helper script is provided at `skills/clawprint-verify/clawprint-challenge.sh` for convenience:

```bash
# Issue a new challenge and display it
./skills/clawprint-verify/clawprint-challenge.sh issue

# Verify an answer
./skills/clawprint-verify/clawprint-challenge.sh verify <challenge_id> <answer>

# Validate a solved challenge server-side
./skills/clawprint-verify/clawprint-challenge.sh validate <challenge_id>
```

## Important Notes

- Each challenge can only be solved once. Replaying a solved challenge returns HTTP 410.
- Speed challenges have very tight time limits (50-500ms). The clock starts when the challenge is issued by the server, so network latency counts.
- Pattern challenges have longer limits (2-10s) but require processing large grids.
- Always validate server-side with your secret key before trusting a result. The verify endpoint confirms the answer is correct, but the validate endpoint confirms it was legitimately solved through your configuration.
- Never share your `CLAWPRINT_SECRET_KEY`. The `CLAWPRINT_SITE_KEY` is safe to expose publicly.

## Failure Reasons

| Reason | Meaning |
|---|---|
| `Too slow: Xms exceeds Yms limit` | Answer was correct but submitted after the time limit |
| `Incorrect answer` | The computed answer was wrong |
| `Challenge not found` | Invalid challenge ID |
| `Challenge already solved` | The challenge was already used (replay attempt) |
