#!/usr/bin/env python3
# API Testing Service - Main Script

import os
import sys
from pathlib import Path
from datetime import datetime
import time

SCRIPT_DIR = Path(__file__).parent.resolve()
OUTPUT_DIR = SCRIPT_DIR / "output"

def test_api_endpoint(url, method="GET", auth=False):
    """测试 API 端点"""
    try:
        import requests

        report = f"# API Test Report: {url}\n\n"
        report += f"**Generated:** {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}\n\n"
        report += f"## Test Configuration\n\n"
        report += f"- URL: {url}\n"
        report += f"- Method: {method}\n"
        report += f"- Authentication: {'Yes' if auth else 'No'}\n\n"

        headers = {"User-Agent": "OpenClaw-AI-Test"}
        if auth:
            headers["Authorization"] = "Bearer test_token"

        report += f"## Test Results\n\n"

        start_time = time.time()
        try:
            response = requests.request(method, url, headers=headers, timeout=10)
            end_time = time.time()

            report += f"- Status Code: {response.status_code}\n"
            report += f"- Response Time: {end_time - start_time:.2f}s\n"
            report += f"- Content Length: {len(response.text)} bytes\n\n"

            # Check status code
            if response.status_code == 200:
                report += f"✅ API is working correctly\n\n"
            elif response.status_code >= 400:
                report += f"⚠️ API returned error status: {response.status_code}\n\n"
            else:
                report += f"ℹ️ API returned warning status: {response.status_code}\n\n"

            # Extract content preview
            preview = response.text[:200] if response.text else "No content"
            report += f"**Response Preview:**\n```\n{preview}\n```\n\n"

        except requests.exceptions.RequestException as e:
            end_time = time.time()
            report += f"- ❌ Request failed: {str(e)}\n"
            report += f"- Response Time: {end_time - start_time:.2f}s\n\n"

        report += f"## Test Summary\n\n"
        report += f"**Status:** {'Passed' if response.status_code == 200 else 'Failed'}\n"
        report += f"**Latency:** {end_time - start_time:.2f}s\n\n"

        report += "---\n\n"
        report += f"*Generated by API Test Service*\n"
        report += f"*Pricing: \$5-15 per test*"

        output_file = OUTPUT_DIR / f"api_test_{datetime.now().strftime('%Y%m%d_%H%M%S')}.md"
        OUTPUT_DIR.mkdir(parents=True, exist_ok=True)
        output_file.write_text(report, encoding='utf-8')

        return str(output_file)
    except Exception as e:
        return f"Error: {str(e)}"

def main():
    if len(sys.argv) < 3:
        print("Usage: python3 main.py <action> <url> [options]")
        print("\nActions:")
        print("  test    <url> [method] [auth] - Test API endpoint")
        print("\nExamples:")
        print("  python3 main.py test https://api.example.com")
        print("  python3 main.py test https://api.example.com/users POST")
        print("  python3 main.py test https://api.example.com/login POST auth")
        sys.exit(1)

    action = sys.argv[1]
    url = sys.argv[2]
    method = sys.argv[3] if len(sys.argv) > 3 else "GET"
    auth = len(sys.argv) > 4 and sys.argv[4].lower() == "auth"

    if action == "test":
        result = test_api_endpoint(url, method, auth)
        print(f"Test saved to: {result}")
    else:
        print(f"Unknown action: {action}")
        sys.exit(1)

if __name__ == "__main__":
    main()
