# AI Agent Tools - Python Utility Library for AI Agents

## 📖 Overview

This library provides ready-to-use Python functions that AI agents can leverage to perform various tasks including file operations, text analysis, data transformation, memory management, and validation.

## ⚡ Quick Start

### Installation

#### Method 1: Clone from GitHub
```bash
git clone https://github.com/cerbug45/ai-agent-tools.git
cd ai-agent-tools
```

#### Method 2: Direct Download
```bash
wget https://raw.githubusercontent.com/cerbug45/ai-agent-tools/main/ai_agent_tools.py
```

#### Method 3: Copy-Paste
Simply copy the `ai_agent_tools.py` file into your project directory.

### Requirements
- Python 3.7 or higher
- No external dependencies (uses only standard library)

## 🛠️ Available Tools

### 1. FileTools - File Operations

Operations for reading, writing, and managing files.

**Available Methods:**

```python
from ai_agent_tools import FileTools

# Read a file
content = FileTools.read_file("path/to/file.txt")

# Write to a file
FileTools.write_file("path/to/file.txt", "Hello World!")

# List files in directory
files = FileTools.list_files(".", extension=".py")

# Check if file exists
exists = FileTools.file_exists("path/to/file.txt")
```

**Use Cases:**
- Reading configuration files
- Saving agent outputs
- Listing available resources
- Checking file existence before operations

---

### 2. TextTools - Text Processing

Extract information and process text data.

**Available Methods:**

```python
from ai_agent_tools import TextTools

text = "Contact: john@example.com, phone: 0532 123 45 67"

# Extract emails
emails = TextTools.extract_emails(text)
# Output: ['john@example.com']

# Extract URLs
urls = TextTools.extract_urls("Visit https://example.com")
# Output: ['https://example.com']

# Extract phone numbers
phones = TextTools.extract_phone_numbers(text)
# Output: ['0532 123 45 67']

# Count words
count = TextTools.word_count("Hello world from AI")
# Output: 4

# Summarize text
summary = TextTools.summarize_text("Long text here...", max_length=50)

# Clean whitespace
clean = TextTools.clean_whitespace("Too   many    spaces")
# Output: "Too many spaces"
```

**Use Cases:**
- Extracting contact information from documents
- Cleaning and formatting text
- Text summarization
- Data extraction from unstructured text

---

### 3. DataTools - Data Transformation

Convert between different data formats.

**Available Methods:**

```python
from ai_agent_tools import DataTools

# Save data as JSON
data = {"name": "Alice", "age": 30}
DataTools.save_json(data, "output.json")

# Load JSON file
loaded_data = DataTools.load_json("output.json")

# Convert CSV text to dictionary list
csv_text = """name,age,city
Alice,30,New York
Bob,25,London"""
data_list = DataTools.csv_to_dict(csv_text)
# Output: [{'name': 'Alice', 'age': '30', 'city': 'New York'}, ...]

# Convert dictionary list to CSV
data = [
    {"name": "Alice", "age": 30},
    {"name": "Bob", "age": 25}
]
csv = DataTools.dict_to_csv(data)
```

**Use Cases:**
- Saving structured data
- Converting between formats
- Processing API responses
- Generating reports

---

### 4. UtilityTools - General Utilities

Helper functions for common operations.

**Available Methods:**

```python
from ai_agent_tools import UtilityTools

# Get current timestamp
timestamp = UtilityTools.get_timestamp()
# Output: "2026-02-15 14:30:25"

# Generate unique ID from text
id = UtilityTools.generate_id("user_john_doe")
# Output: "a3f5b2c1"

# Calculate percentage
percent = UtilityTools.calculate_percentage(25, 100)
# Output: 25.0

# Safe division (no divide by zero error)
result = UtilityTools.safe_divide(10, 0, default=0.0)
# Output: 0.0
```

**Use Cases:**
- Timestamping events
- Generating unique identifiers
- Safe mathematical operations
- Data analysis calculations

---

### 5. MemoryTools - Memory Management

Store and retrieve data during agent execution.

**Available Methods:**

```python
from ai_agent_tools import MemoryTools

# Initialize memory
memory = MemoryTools()

# Store a value
memory.store("user_name", "Alice")
memory.store("session_id", "abc123")

# Retrieve a value
name = memory.retrieve("user_name")
# Output: "Alice"

# List all keys
keys = memory.list_keys()
# Output: ["user_name", "session_id"]

# Delete a value
memory.delete("session_id")

# Clear all memory
memory.clear()
```

**Use Cases:**
- Maintaining conversation context
- Storing intermediate results
- Session management
- Caching computed values

---

### 6. ValidationTools - Data Validation

Validate different types of data.

**Available Methods:**

```python
from ai_agent_tools import ValidationTools

# Validate email
is_valid = ValidationTools.is_valid_email("user@example.com")
# Output: True

# Validate URL
is_valid = ValidationTools.is_valid_url("https://example.com")
# Output: True

# Validate phone number (Turkish format)
is_valid = ValidationTools.is_valid_phone("0532 123 45 67")
# Output: True
```

**Use Cases:**
- Input validation
- Data quality checks
- Form validation
- Pre-processing data

---

## 💡 Complete Usage Example

```python
from ai_agent_tools import (
    FileTools, TextTools, DataTools, 
    UtilityTools, MemoryTools, ValidationTools
)

# Initialize memory for session
memory = MemoryTools()

# Read input file
text = FileTools.read_file("contacts.txt")

# Extract information
emails = TextTools.extract_emails(text)
phones = TextTools.extract_phone_numbers(text)

# Validate extracted data
valid_emails = [e for e in emails if ValidationTools.is_valid_email(e)]
valid_phones = [p for p in phones if ValidationTools.is_valid_phone(p)]

# Create structured data
contacts = []
for i, (email, phone) in enumerate(zip(valid_emails, valid_phones)):
    contact = {
        "id": UtilityTools.generate_id(f"contact_{i}"),
        "email": email,
        "phone": phone,
        "timestamp": UtilityTools.get_timestamp()
    }
    contacts.append(contact)

# Save results
DataTools.save_json(contacts, "output/contacts.json")

# Store in memory
memory.store("total_contacts", len(contacts))
memory.store("last_processed", UtilityTools.get_timestamp())

print(f"Processed {len(contacts)} contacts")
print(f"Saved to: output/contacts.json")
```

## 🎯 Best Practices

### 1. Error Handling
Always wrap file operations in try-except blocks:

```python
try:
    content = FileTools.read_file("data.txt")
    # Process content
except Exception as e:
    print(f"Error reading file: {e}")
```

### 2. Memory Management
Clear memory when no longer needed:

```python
memory = MemoryTools()
# ... use memory ...
memory.clear()  # Clean up
```

### 3. Data Validation
Always validate data before processing:

```python
if ValidationTools.is_valid_email(email):
    # Process email
    pass
else:
    print(f"Invalid email: {email}")
```

### 4. Path Handling
Use absolute paths or ensure working directory is correct:

```python
import os

base_dir = os.path.dirname(__file__)
filepath = os.path.join(base_dir, "data", "file.txt")
content = FileTools.read_file(filepath)
```

## 🔧 Advanced Usage

### Chaining Operations

```python
# Read -> Process -> Validate -> Save pipeline
text = FileTools.read_file("input.txt")
cleaned = TextTools.clean_whitespace(text)
emails = TextTools.extract_emails(cleaned)
valid = [e for e in emails if ValidationTools.is_valid_email(e)]
DataTools.save_json({"emails": valid}, "output.json")
```

### Creating Custom Workflows

```python
class DataProcessor:
    def __init__(self):
        self.memory = MemoryTools()
        
    def process_document(self, filepath):
        # Read
        text = FileTools.read_file(filepath)
        
        # Extract
        emails = TextTools.extract_emails(text)
        urls = TextTools.extract_urls(text)
        
        # Store results
        self.memory.store("emails", emails)
        self.memory.store("urls", urls)
        
        # Generate report
        report = {
            "timestamp": UtilityTools.get_timestamp(),
            "file": filepath,
            "emails_found": len(emails),
            "urls_found": len(urls)
        }
        
        return report
```

## 📦 Integration with AI Agents

### Example: LangChain Integration

```python
from langchain.tools import Tool
from ai_agent_tools import FileTools, TextTools

def create_file_reader_tool():
    return Tool(
        name="ReadFile",
        func=FileTools.read_file,
        description="Read contents of a file"
    )

def create_email_extractor_tool():
    return Tool(
        name="ExtractEmails",
        func=TextTools.extract_emails,
        description="Extract email addresses from text"
    )

tools = [create_file_reader_tool(), create_email_extractor_tool()]
```

### Example: OpenAI Function Calling

```python
tools = [
    {
        "type": "function",
        "function": {
            "name": "read_file",
            "description": "Read a file and return its contents",
            "parameters": {
                "type": "object",
                "properties": {
                    "filepath": {
                        "type": "string",
                        "description": "Path to the file"
                    }
                },
                "required": ["filepath"]
            }
        }
    }
]

# In your agent loop
def execute_function(name, arguments):
    if name == "read_file":
        return FileTools.read_file(arguments["filepath"])
```

## 🧪 Testing

Run the built-in test suite:

```bash
python ai_agent_tools.py
```

Expected output:
```
=== AI Ajanları İçin Araçlar Kütüphanesi ===

1. Dosya Araçları:
   Okunan içerik: Merhaba AI Ajanı!

2. Metin Araçları:
   Bulunan emailler: ['ali@example.com']
   Bulunan telefonlar: ['0532 123 45 67']

3. Veri Araçları:
   CSV çıktısı:
   isim,yaş
   Ali,25
   Ayşe,30

...

✓ Tüm araçlar test edildi!
```

## 🤝 Contributing

Contributions are welcome! To contribute:

1. Fork the repository
2. Create a feature branch: `git checkout -b feature/new-tool`
3. Commit your changes: `git commit -am 'Add new tool'`
4. Push to the branch: `git push origin feature/new-tool`
5. Submit a pull request

## 📝 License

This project is open source and available under the MIT License.

## 👤 Author

**GitHub:** [@cerbug45](https://github.com/cerbug45)

## 🐛 Issues & Support

Found a bug or need help? Please open an issue on GitHub:
https://github.com/cerbug45/ai-agent-tools/issues

## 📚 Additional Resources

- [Python Documentation](https://docs.python.org/3/)
- [Regular Expressions Guide](https://docs.python.org/3/howto/regex.html)
- [JSON Format Specification](https://www.json.org/)

## 🔄 Version History

### v1.0.0 (2026-02-15)
- Initial release
- 6 tool categories
- 25+ utility functions
- Full documentation
- Test suite included

---

**Happy Coding! 🚀**
