#!/bin/bash

# bump-changed-plugins.sh
# Intelligently bump versions for only the plugins with changes
# Detects changed plugins via git diff and version history
# Usage: ./bump-changed-plugins.sh [major|minor|patch]
# Default: patch

set -euo pipefail

BUMP_LEVEL="${1:-patch}"
MARKETPLACE=".claude-plugin/marketplace.json"
REPO_ROOT=$(git rev-parse --show-toplevel)

# Validate bump level
case "$BUMP_LEVEL" in
    major|minor|patch) ;;
    *) echo "Error: Invalid bump level '$BUMP_LEVEL' (use: major, minor, or patch)"; exit 1 ;;
esac

echo "🔍 Detecting changed plugins..."
echo ""

# Find plugins with changes since last commit
changed_plugins=()

while IFS= read -r plugin_dir; do
    [ -z "$plugin_dir" ] && continue
    plugin_name=$(basename "$plugin_dir")

    # Check if plugin has uncommitted changes or unpushed commits
    if git diff --quiet "$plugin_dir" 2>/dev/null; then
        if git diff --cached --quiet "$plugin_dir" 2>/dev/null; then
            # No changes, skip
            continue
        fi
    fi

    changed_plugins+=("$plugin_name")
done < <(find "$REPO_ROOT/plugins" -maxdepth 1 -type d ! -name plugins)

# If no uncommitted changes, check for commits since main branch
if [ ${#changed_plugins[@]} -eq 0 ]; then
    echo "No uncommitted changes detected. Checking for commits since main..."

    # Compare against main or origin/main
    compare_branch="main"
    if ! git rev-parse --verify "$compare_branch" &>/dev/null; then
        compare_branch="origin/main"
    fi

    while IFS= read -r plugin_dir; do
        [ -z "$plugin_dir" ] && continue
        plugin_name=$(basename "$plugin_dir")

        if git diff "$compare_branch..." --name-only -- "$plugin_dir" 2>/dev/null | grep -q .; then
            changed_plugins+=("$plugin_name")
        fi
    done < <(find "$REPO_ROOT/plugins" -maxdepth 1 -type d ! -name plugins)
fi

if [ ${#changed_plugins[@]} -eq 0 ]; then
    echo "ℹ️  No changed plugins detected."
    echo "Hint: Stage or commit changes in plugin directories to auto-detect them."
    exit 0
fi

echo "Found ${#changed_plugins[@]} changed plugin(s):"
for plugin in "${changed_plugins[@]}"; do
    echo "  • $plugin"
done
echo ""

# Bump each changed plugin
bumped_count=0

for plugin in "${changed_plugins[@]}"; do
    plugin_json="$REPO_ROOT/plugins/${plugin}/.claude-plugin/plugin.json"

    if [ ! -f "$plugin_json" ]; then
        echo "⚠️  Skipping $plugin (plugin.json not found)"
        continue
    fi

    # Get current version
    current=$(jq -r '.version // "0.0.0"' "$plugin_json")

    # Parse version parts
    IFS='.' read -r major minor patch <<< "$current"
    patch=${patch:-0}

    # Calculate new version
    case "$BUMP_LEVEL" in
        major) new="$((major + 1)).0.0" ;;
        minor) new="${major}.$((minor + 1)).0" ;;
        patch) new="${major}.${minor}.$((patch + 1))" ;;
    esac

    echo "📦 $plugin: $current → $new"

    # Update plugin's plugin.json
    if ! jq --arg version "$new" '.version = $version' "$plugin_json" > "${plugin_json}.tmp"; then
        echo "❌ Failed to update $plugin_json"
        rm -f "${plugin_json}.tmp"
        continue
    fi
    mv "${plugin_json}.tmp" "$plugin_json"

    # Update marketplace.json if it exists and has this plugin
    if [ -f "$MARKETPLACE" ]; then
        if jq -e --arg name "$plugin" '.plugins[] | select(.name == $name)' "$MARKETPLACE" > /dev/null 2>&1; then
            if ! jq --arg name "$plugin" --arg version "$new" \
                '(.plugins[] | select(.name == $name) | .version) = $version' \
                "$MARKETPLACE" > "${MARKETPLACE}.tmp"; then
                echo "❌ Failed to update $MARKETPLACE"
                rm -f "${MARKETPLACE}.tmp"
                continue
            fi
            mv "${MARKETPLACE}.tmp" "$MARKETPLACE"
        fi
    fi

    bumped_count=$((bumped_count + 1))
done

echo ""
echo "✅ Bumped $bumped_count plugin(s) to $BUMP_LEVEL"
echo ""
echo "Next: git add -A && git commit -m 'chore: bump versions for changed plugins'"
