#!/usr/bin/env bash
set -euo pipefail

PROJECT_DIR="${PROJECT_DIR:-$HOME/agentic_paper_digest}"
REPO_URL="https://github.com/matanle51/agentic_paper_digest"
ZIP_URL="https://github.com/matanle51/agentic_paper_digest/archive/refs/heads/main.zip"

if command -v python3 >/dev/null 2>&1; then
  PY_BIN="python3"
elif command -v python >/dev/null 2>&1; then
  PY_BIN="python"
else
  echo "Python not found. Install Python 3 and retry."
  exit 1
fi

if command -v git >/dev/null 2>&1; then
  if [ -d "$PROJECT_DIR/.git" ]; then
    echo ">> Updating repo in $PROJECT_DIR"
    git -C "$PROJECT_DIR" pull --ff-only
  else
    echo ">> Cloning repo into $PROJECT_DIR"
    git clone "$REPO_URL" "$PROJECT_DIR"
  fi
else
  if [ -d "$PROJECT_DIR" ] && [ "$(ls -A "$PROJECT_DIR")" ]; then
    echo ">> Git not available. Using existing contents in $PROJECT_DIR"
  else
    echo ">> Git not available. Downloading zip to $PROJECT_DIR"
    mkdir -p "$PROJECT_DIR"
    tmp_zip="$(mktemp -t agentic_paper_digest.XXXXXX).zip"
    if command -v curl >/dev/null 2>&1; then
      curl -L "$ZIP_URL" -o "$tmp_zip"
    elif command -v wget >/dev/null 2>&1; then
      wget -O "$tmp_zip" "$ZIP_URL"
    else
      "$PY_BIN" - "$ZIP_URL" "$tmp_zip" <<'PY'
import sys
import urllib.request
url, out = sys.argv[1], sys.argv[2]
urllib.request.urlretrieve(url, out)
PY
    fi

    tmp_dir="$(mktemp -d)"
    if command -v unzip >/dev/null 2>&1; then
      unzip -q "$tmp_zip" -d "$tmp_dir"
    else
      "$PY_BIN" - "$tmp_zip" "$tmp_dir" <<'PY'
import sys
import zipfile
zipfile.ZipFile(sys.argv[1]).extractall(sys.argv[2])
PY
    fi

    src_dir="$tmp_dir/agentic_paper_digest-main"
    if [ ! -d "$src_dir" ]; then
      echo "Downloaded archive did not contain expected folder: $src_dir"
      exit 1
    fi

    cp -R "$src_dir/." "$PROJECT_DIR/"
    rm -rf "$tmp_zip" "$tmp_dir"
  fi
fi

cd "$PROJECT_DIR"

if [ ! -d ".venv" ]; then
  echo ">> Creating virtualenv"
  "$PY_BIN" -m venv .venv
fi

# shellcheck disable=SC1091
source .venv/bin/activate

echo ">> Installing Python deps"
pip install -r requirements.txt

if [ ! -f ".env" ] && [ -f ".env.example" ]; then
  cp .env.example .env
  echo ">> Created .env from .env.example"
fi

echo ">> Done. Edit .env, then run scripts/run_cli.sh or scripts/run_api.sh"
