#!/usr/bin/env bash
# sms-notify.sh - Send SMS notifications via Twilio
# Part of agentic-calling skill for Clawdbot

set -euo pipefail

# Configuration
CONFIG_FILE="${HOME}/.clawdbot/twilio-config.json"
TWILIO_ACCOUNT_SID="${TWILIO_ACCOUNT_SID:-}"
TWILIO_AUTH_TOKEN="${TWILIO_AUTH_TOKEN:-}"
TWILIO_PHONE_NUMBER="${TWILIO_PHONE_NUMBER:-}"

# Parameters
TO_NUMBER=""
MESSAGE=""
MEDIA_URL=""

# Load config
if [[ -f "$CONFIG_FILE" ]]; then
  TWILIO_ACCOUNT_SID=$(jq -r '.accountSid // empty' "$CONFIG_FILE")
  TWILIO_AUTH_TOKEN=$(jq -r '.authToken // empty' "$CONFIG_FILE")
  TWILIO_PHONE_NUMBER=$(jq -r '.phoneNumber // empty' "$CONFIG_FILE")
fi

# Parse arguments
while [[ $# -gt 0 ]]; do
  case $1 in
    --to)
      TO_NUMBER="$2"
      shift 2
      ;;
    --message)
      MESSAGE="$2"
      shift 2
      ;;
    --media)
      MEDIA_URL="$2"
      shift 2
      ;;
    --help)
      echo "Usage: $0 --to <number> --message <text> [--media <url>]"
      echo ""
      echo "Options:"
      echo "  --to <number>      Destination phone number (E.164 format)"
      echo "  --message <text>   Message to send"
      echo "  --media <url>      Media URL for MMS (optional)"
      exit 0
      ;;
    *)
      echo "Unknown option: $1"
      exit 1
      ;;
  esac
done

# Validate
if [[ -z "$TWILIO_ACCOUNT_SID" ]] || [[ -z "$TWILIO_AUTH_TOKEN" ]] || [[ -z "$TWILIO_PHONE_NUMBER" ]]; then
  echo "Error: Twilio credentials not configured"
  exit 1
fi

if [[ -z "$TO_NUMBER" ]] || [[ -z "$MESSAGE" ]]; then
  echo "Error: --to and --message are required"
  exit 1
fi

# Build request
API_URL="https://api.twilio.com/2010-04-01/Accounts/${TWILIO_ACCOUNT_SID}/Messages.json"
FORM_DATA="To=${TO_NUMBER}&From=${TWILIO_PHONE_NUMBER}&Body=${MESSAGE}"

if [[ -n "$MEDIA_URL" ]]; then
  FORM_DATA="${FORM_DATA}&MediaUrl=${MEDIA_URL}"
fi

# Send SMS
echo "Sending SMS to ${TO_NUMBER}..."
RESPONSE=$(curl -s -X POST "$API_URL" \
  -u "${TWILIO_ACCOUNT_SID}:${TWILIO_AUTH_TOKEN}" \
  -d "$FORM_DATA")

MESSAGE_SID=$(echo "$RESPONSE" | jq -r '.sid // empty')
STATUS=$(echo "$RESPONSE" | jq -r '.status // empty')
ERROR_MESSAGE=$(echo "$RESPONSE" | jq -r '.message // empty')

if [[ -n "$MESSAGE_SID" ]]; then
  echo "✅ SMS sent successfully!"
  echo "Message SID: $MESSAGE_SID"
  echo "Status: $STATUS"
else
  echo "❌ SMS failed:"
  echo "$ERROR_MESSAGE"
  exit 1
fi
