#!/usr/bin/env bash
# make-call.sh - Make outbound phone calls using Twilio
# Part of agentic-calling skill for Clawdbot

set -euo pipefail

# Configuration
CONFIG_FILE="${HOME}/.clawdbot/twilio-config.json"
TWILIO_ACCOUNT_SID="${TWILIO_ACCOUNT_SID:-}"
TWILIO_AUTH_TOKEN="${TWILIO_AUTH_TOKEN:-}"
TWILIO_PHONE_NUMBER="${TWILIO_PHONE_NUMBER:-}"

# Default parameters
TO_NUMBER=""
MESSAGE=""
VOICE="Polly.Joanna"
RECORD="false"
CALLBACK_URL=""
TIMEOUT="30"
TRANSCRIBE="false"
URGENT="false"

# Load config from file if exists
if [[ -f "$CONFIG_FILE" ]]; then
  TWILIO_ACCOUNT_SID=$(jq -r '.accountSid // empty' "$CONFIG_FILE")
  TWILIO_AUTH_TOKEN=$(jq -r '.authToken // empty' "$CONFIG_FILE")
  TWILIO_PHONE_NUMBER=$(jq -r '.phoneNumber // empty' "$CONFIG_FILE")
fi

# Parse arguments
while [[ $# -gt 0 ]]; do
  case $1 in
    --to)
      TO_NUMBER="$2"
      shift 2
      ;;
    --message)
      MESSAGE="$2"
      shift 2
      ;;
    --voice)
      VOICE="$2"
      shift 2
      ;;
    --record)
      RECORD="$2"
      shift 2
      ;;
    --callback)
      CALLBACK_URL="$2"
      shift 2
      ;;
    --timeout)
      TIMEOUT="$2"
      shift 2
      ;;
    --transcribe)
      TRANSCRIBE="$2"
      shift 2
      ;;
    --urgent)
      URGENT="$2"
      shift 2
      ;;
    --help)
      echo "Usage: $0 --to <number> --message <text> [options]"
      echo ""
      echo "Options:"
      echo "  --to <number>       Destination phone number (E.164 format)"
      echo "  --message <text>    Message to speak"
      echo "  --voice <voice>     Voice to use (default: Polly.Joanna)"
      echo "  --record <bool>     Record the call (default: false)"
      echo "  --callback <url>    Status callback URL"
      echo "  --timeout <sec>     Ring timeout in seconds (default: 30)"
      echo "  --transcribe <bool> Transcribe recording (default: false)"
      echo "  --urgent <bool>     Urgent priority (default: false)"
      exit 0
      ;;
    *)
      echo "Unknown option: $1"
      exit 1
      ;;
  esac
done

# Validate required parameters
if [[ -z "$TWILIO_ACCOUNT_SID" ]] || [[ -z "$TWILIO_AUTH_TOKEN" ]] || [[ -z "$TWILIO_PHONE_NUMBER" ]]; then
  echo "Error: Twilio credentials not configured"
  echo "Set TWILIO_ACCOUNT_SID, TWILIO_AUTH_TOKEN, and TWILIO_PHONE_NUMBER environment variables"
  echo "Or create config file at: $CONFIG_FILE"
  exit 1
fi

if [[ -z "$TO_NUMBER" ]]; then
  echo "Error: --to parameter is required"
  exit 1
fi

if [[ -z "$MESSAGE" ]]; then
  echo "Error: --message parameter is required"
  exit 1
fi

# Construct TwiML
TWIML="<?xml version=\"1.0\" encoding=\"UTF-8\"?><Response><Say voice=\"$VOICE\">$MESSAGE</Say>"
if [[ "$URGENT" == "true" ]]; then
  TWIML="${TWIML}<Say voice=\"$VOICE\">This is an urgent message. Please respond immediately.</Say>"
fi
TWIML="${TWIML}</Response>"

# URL encode TwiML
TWIML_ENCODED=$(python3 -c "import urllib.parse; print(urllib.parse.quote('''$TWIML'''))")

# Build API request
API_URL="https://api.twilio.com/2010-04-01/Accounts/${TWILIO_ACCOUNT_SID}/Calls.json"

# Build form data
FORM_DATA="To=${TO_NUMBER}&From=${TWILIO_PHONE_NUMBER}&Twiml=${TWIML_ENCODED}&Timeout=${TIMEOUT}"

if [[ "$RECORD" == "true" ]]; then
  FORM_DATA="${FORM_DATA}&Record=true"
  if [[ "$TRANSCRIBE" == "true" ]]; then
    FORM_DATA="${FORM_DATA}&RecordingStatusCallback=${CALLBACK_URL}/recording&Transcribe=true"
  fi
fi

if [[ -n "$CALLBACK_URL" ]]; then
  FORM_DATA="${FORM_DATA}&StatusCallback=${CALLBACK_URL}"
fi

# Make API call
echo "Making call to ${TO_NUMBER}..."
RESPONSE=$(curl -s -X POST "$API_URL" \
  -u "${TWILIO_ACCOUNT_SID}:${TWILIO_AUTH_TOKEN}" \
  -d "$FORM_DATA")

# Parse response
CALL_SID=$(echo "$RESPONSE" | jq -r '.sid // empty')
STATUS=$(echo "$RESPONSE" | jq -r '.status // empty')
ERROR_MESSAGE=$(echo "$RESPONSE" | jq -r '.message // empty')

if [[ -n "$CALL_SID" ]]; then
  echo "✅ Call initiated successfully!"
  echo "Call SID: $CALL_SID"
  echo "Status: $STATUS"
  echo ""
  echo "To check status:"
  echo "  ./call-status.sh --sid $CALL_SID"
else
  echo "❌ Call failed:"
  echo "$ERROR_MESSAGE"
  exit 1
fi
