#!/usr/bin/env bash
# call-status.sh - Check status of Twilio calls
# Part of agentic-calling skill for Clawdbot

set -euo pipefail

# Configuration
CONFIG_FILE="${HOME}/.clawdbot/twilio-config.json"
TWILIO_ACCOUNT_SID="${TWILIO_ACCOUNT_SID:-}"
TWILIO_AUTH_TOKEN="${TWILIO_AUTH_TOKEN:-}"

# Parameters
CALL_SID=""
LIST_CALLS="false"
LIMIT=10
DOWNLOAD_RECORDING="false"
GET_TRANSCRIPT="false"
OUTPUT_FILE=""

# Load config
if [[ -f "$CONFIG_FILE" ]]; then
  TWILIO_ACCOUNT_SID=$(jq -r '.accountSid // empty' "$CONFIG_FILE")
  TWILIO_AUTH_TOKEN=$(jq -r '.authToken // empty' "$CONFIG_FILE")
fi

# Parse arguments
while [[ $# -gt 0 ]]; do
  case $1 in
    --sid)
      CALL_SID="$2"
      shift 2
      ;;
    --list)
      LIST_CALLS="true"
      shift
      ;;
    --limit)
      LIMIT="$2"
      shift 2
      ;;
    --download-recording)
      DOWNLOAD_RECORDING="true"
      shift
      ;;
    --get-transcript)
      GET_TRANSCRIPT="true"
      shift
      ;;
    --output)
      OUTPUT_FILE="$2"
      shift 2
      ;;
    --help)
      echo "Usage: $0 [--sid <call_sid>] [--list] [options]"
      echo ""
      echo "Options:"
      echo "  --sid <call_sid>        Get status of specific call"
      echo "  --list                  List recent calls"
      echo "  --limit <num>           Number of calls to list (default: 10)"
      echo "  --download-recording    Download call recording"
      echo "  --get-transcript        Get call transcription"
      echo "  --output <file>         Output file for recording"
      exit 0
      ;;
    *)
      echo "Unknown option: $1"
      exit 1
      ;;
  esac
done

# Validate credentials
if [[ -z "$TWILIO_ACCOUNT_SID" ]] || [[ -z "$TWILIO_AUTH_TOKEN" ]]; then
  echo "Error: Twilio credentials not configured"
  exit 1
fi

if [[ "$LIST_CALLS" == "true" ]]; then
  # List recent calls
  API_URL="https://api.twilio.com/2010-04-01/Accounts/${TWILIO_ACCOUNT_SID}/Calls.json?PageSize=${LIMIT}"
  
  RESPONSE=$(curl -s -X GET "$API_URL" \
    -u "${TWILIO_ACCOUNT_SID}:${TWILIO_AUTH_TOKEN}")
  
  echo "$RESPONSE" | jq -r '.calls[] | "\(.sid)\t\(.from)\t\(.to)\t\(.status)\t\(.start_time)\t\(.duration)s"' \
    | column -t -s $'\t' -N "SID,From,To,Status,Time,Duration"
  
elif [[ -n "$CALL_SID" ]]; then
  # Get specific call status
  API_URL="https://api.twilio.com/2010-04-01/Accounts/${TWILIO_ACCOUNT_SID}/Calls/${CALL_SID}.json"
  
  RESPONSE=$(curl -s -X GET "$API_URL" \
    -u "${TWILIO_ACCOUNT_SID}:${TWILIO_AUTH_TOKEN}")
  
  echo "Call Details:"
  echo "$RESPONSE" | jq '{"SID": .sid, "From": .from, "To": .to, "Status": .status, "Duration": .duration, "Price": .price, "Direction": .direction}'
  
  if [[ "$DOWNLOAD_RECORDING" == "true" ]]; then
    # Get recordings for this call
    REC_URL="https://api.twilio.com/2010-04-01/Accounts/${TWILIO_ACCOUNT_SID}/Recordings.json?CallSid=${CALL_SID}"
    
    REC_RESPONSE=$(curl -s -X GET "$REC_URL" \
      -u "${TWILIO_ACCOUNT_SID}:${TWILIO_AUTH_TOKEN}")
    
    REC_SID=$(echo "$REC_RESPONSE" | jq -r '.recordings[0].sid // empty')
    
    if [[ -n "$REC_SID" ]]; then
      OUTPUT_FILE="${OUTPUT_FILE:-${CALL_SID}.mp3}"
      DOWNLOAD_URL="https://api.twilio.com/2010-04-01/Accounts/${TWILIO_ACCOUNT_SID}/Recordings/${REC_SID}.mp3"
      
      curl -s -o "$OUTPUT_FILE" -u "${TWILIO_ACCOUNT_SID}:${TWILIO_AUTH_TOKEN}" "$DOWNLOAD_URL"
      echo "Recording downloaded to: $OUTPUT_FILE"
    else
      echo "No recording found for this call"
    fi
  fi
  
  if [[ "$GET_TRANSCRIPT" == "true" ]]; then
    # Get transcription
    TRANS_URL="https://api.twilio.com/2010-04-01/Accounts/${TWILIO_ACCOUNT_SID}/Transcriptions.json?RecordingSid=${REC_SID}"
    
    TRANS_RESPONSE=$(curl -s -X GET "$TRANS_URL" \
      -u "${TWILIO_ACCOUNT_SID}:${TWILIO_AUTH_TOKEN}")
    
    echo "Transcription:"
    echo "$TRANS_RESPONSE" | jq -r '.transcriptions[0].transcription_text // "No transcription available"'
  fi
else
  echo "Error: Provide --sid or --list"
  exit 1
fi
