# Agentic Calling Skill

**Enable AI agents to make and receive phone calls autonomously using Twilio.**

## Overview

This skill provides a complete toolkit for AI agents to handle phone calls programmatically. Agents can:
- Make outbound calls with custom voice messages
- Receive inbound calls and respond dynamically
- Convert text to speech for natural conversations
- Transcribe caller speech to text
- Handle call routing and forwarding
- Manage voicemail and recordings

## Prerequisites

1. **Twilio Account**: Sign up at [twilio.com](https://www.twilio.com)
2. **Twilio Phone Number**: Purchase a number with Voice capabilities
3. **Twilio Credentials**: Account SID and Auth Token

## Quick Start

### 1. Configure Credentials

Create a file at `~/.clawdbot/twilio-config.json`:

```json
{
  "accountSid": "YOUR_ACCOUNT_SID",
  "authToken": "YOUR_AUTH_TOKEN",
  "phoneNumber": "+1XXXXXXXXXX"
}
```

Or set environment variables:

```bash
export TWILIO_ACCOUNT_SID="YOUR_ACCOUNT_SID"
export TWILIO_AUTH_TOKEN="YOUR_AUTH_TOKEN"
export TWILIO_PHONE_NUMBER="+1XXXXXXXXXX"
```

### 2. Make Your First Call

```bash
./scripts/make-call.sh --to "+15551234567" --message "Hello! This is your AI assistant calling."
```

### 3. Set Up Inbound Call Handling

```bash
./scripts/setup-webhook.sh --url "https://your-server.com/voice"
```

## Core Scripts

### `make-call.sh` - Make Outbound Calls

Make a phone call with a text-to-speech message:

```bash
# Simple call with message
./scripts/make-call.sh --to "+15551234567" --message "Hello from your AI assistant"

# Call with custom voice
./scripts/make-call.sh --to "+15551234567" --message "Important update" --voice "Polly.Matthew"

# Call with recording
./scripts/make-call.sh --to "+15551234567" --message "Please hold" --record true

# Call with status callback
./scripts/make-call.sh --to "+15551234567" --message "Hello" --callback "https://your-server.com/status"
```

**Parameters:**
- `--to` (required): Destination phone number (E.164 format)
- `--message` (required): Text to speak
- `--voice` (optional): Voice to use (default: Polly.Joanna)
- `--record` (optional): Record the call (true/false)
- `--callback` (optional): URL for status updates
- `--timeout` (optional): Ring timeout in seconds (default: 30)

### `receive-call.sh` - Handle Inbound Calls

Server script to handle incoming calls with TwiML responses:

```bash
# Start webhook server on port 3000
./scripts/receive-call.sh --port 3000

# Custom greeting
./scripts/receive-call.sh --port 3000 --greeting "Thank you for calling AI Services"

# Forward to another number
./scripts/receive-call.sh --port 3000 --forward "+15559876543"

# Record voicemail
./scripts/receive-call.sh --port 3000 --voicemail true
```

### `sms-notify.sh` - Send SMS Notifications

Send SMS messages (useful for call follow-ups):

```bash
# Simple SMS
./scripts/sms-notify.sh --to "+15551234567" --message "Missed call from AI assistant"

# With media (MMS)
./scripts/sms-notify.sh --to "+15551234567" --message "Summary attached" --media "https://example.com/summary.pdf"
```

### `call-status.sh` - Check Call Status

Monitor active and completed calls:

```bash
# Get status of specific call
./scripts/call-status.sh --sid "CA1234567890abcdef"

# List recent calls
./scripts/call-status.sh --list --limit 10

# Get call recording
./scripts/call-status.sh --sid "CA1234567890abcdef" --download-recording
```

## Advanced Usage

### Custom IVR (Interactive Voice Response)

Create dynamic phone menus:

```bash
./scripts/create-ivr.sh --menu "Press 1 for sales, 2 for support, 3 for emergencies"
```

### Conference Calls

Set up multi-party conference calls:

```bash
# Create conference
./scripts/conference.sh --create --name "Team Standup"

# Add participant
./scripts/conference.sh --add-participant --conference "Team Standup" --number "+15551234567"
```

### Call Recording & Transcription

```bash
# Record and transcribe
./scripts/make-call.sh --to "+15551234567" --message "How can I help?" --record true --transcribe true

# Download recording
./scripts/call-status.sh --sid "CA123..." --download-recording --output "call.mp3"

# Get transcription
./scripts/call-status.sh --sid "CA123..." --get-transcript
```

### Voice Cloning (Experimental)

Use ElevenLabs integration for custom voice:

```bash
# Requires ElevenLabs API key
./scripts/make-call-elevenlabs.sh --to "+15551234567" --message "Hello" --voice-id "YOUR_VOICE_ID"
```

## Integration Patterns

### 1. Appointment Reminders

```bash
#!/bin/bash
# Send appointment reminder calls
while read -r name phone appointment; do
  ./scripts/make-call.sh \
    --to "$phone" \
    --message "Hello $name, this is a reminder about your appointment on $appointment. Press 1 to confirm, 2 to reschedule."
done < appointments.txt
```

### 2. Emergency Alerts

```bash
#!/bin/bash
# Broadcast emergency alert to list
emergency_message="Emergency alert: System outage detected. Team members are working on resolution."

cat on-call-list.txt | while read phone; do
  ./scripts/make-call.sh \
    --to "$phone" \
    --message "$emergency_message" \
    --urgent true &
done
wait
```

### 3. Lead Qualification

```bash
#!/bin/bash
# Call leads and route based on IVR response
./scripts/make-call.sh \
  --to "+15551234567" \
  --message "Thank you for your interest. Press 1 if you'd like to schedule a demo, 2 for pricing information, or 3 to speak with a representative." \
  --callback "https://your-crm.com/lead-response"
```

## Voice Options

Supported voices (Amazon Polly):

**English (US):**
- `Polly.Joanna` (Female, default)
- `Polly.Matthew` (Male)
- `Polly.Ivy` (Female, child)
- `Polly.Joey` (Male)
- `Polly.Kendra` (Female)
- `Polly.Kimberly` (Female)
- `Polly.Salli` (Female)

**English (UK):**
- `Polly.Amy` (Female)
- `Polly.Brian` (Male)
- `Polly.Emma` (Female)

**Other Languages:**
- Spanish: `Polly.Miguel`, `Polly.Penelope`
- French: `Polly.Celine`, `Polly.Mathieu`
- German: `Polly.Hans`, `Polly.Marlene`

## Webhooks & TwiML

### Setting Up Webhooks

Configure your Twilio number to POST to your webhook URL when calls arrive:

```bash
./scripts/configure-number.sh \
  --voice-url "https://your-server.com/voice" \
  --voice-method "POST" \
  --status-callback "https://your-server.com/status"
```

### Example TwiML Response

```xml
<?xml version="1.0" encoding="UTF-8"?>
<Response>
    <Say voice="Polly.Joanna">Hello! Thank you for calling.</Say>
    <Gather numDigits="1" action="/handle-key">
        <Say>Press 1 for sales, 2 for support, or 3 to leave a message.</Say>
    </Gather>
</Response>
```

## Cost Optimization

- **Outbound calls**: ~$0.013/minute (US)
- **Inbound calls**: ~$0.0085/minute (US)
- **SMS**: ~$0.0079/message (US)
- **Phone number**: ~$1.15/month

**Tips:**
- Use regional phone numbers to reduce costs
- Batch calls during off-peak hours
- Keep messages concise to minimize call duration
- Use SMS for simple notifications

## Security Best Practices

1. **Protect Credentials**: Never commit credentials to git
2. **Use HTTPS**: Always use HTTPS for webhooks
3. **Validate Requests**: Verify Twilio signatures on webhooks
4. **Rate Limiting**: Implement rate limits on outbound calls
5. **Logging**: Log all calls for audit trails

## Troubleshooting

### Call Not Connecting

```bash
# Check number formatting (must be E.164)
./scripts/validate-number.sh "+15551234567"

# Test connectivity
./scripts/make-call.sh --to "$TWILIO_PHONE_NUMBER" --message "Test call"
```

### Webhook Not Receiving Calls

```bash
# Test webhook
curl -X POST https://your-server.com/voice \
  -d "Called=+15551234567" \
  -d "From=+15559876543"

# Check Twilio debugger
./scripts/check-logs.sh --recent 10
```

### Audio Quality Issues

```bash
# Use different voice engine
./scripts/make-call.sh --to "+15551234567" --message "Test" --voice "Google.en-US-Neural2-A"

# Adjust speech rate
./scripts/make-call.sh --to "+15551234567" --message "Test" --rate "90%"
```

## Examples

See `examples/` directory for complete use cases:

- `examples/appointment-reminder.sh` - Automated appointment reminders
- `examples/emergency-broadcast.sh` - Broadcast emergency alerts
- `examples/ivr-menu.sh` - Interactive voice menu
- `examples/voicemail-transcription.sh` - Voicemail to email
- `examples/two-factor-auth.sh` - Voice-based 2FA

## API Reference

Full Twilio API documentation: https://www.twilio.com/docs/voice

## Support

- GitHub Issues: [Report bugs or request features]
- Twilio Docs: https://www.twilio.com/docs
- Community: https://discord.com/invite/clawd

## License

MIT License - feel free to use in your own projects

## Credits

Created by Kelly Claude (AI Assistant)
Powered by Twilio and Clawdbot
