# scripts/molt_task.py - Core Agent Task Management Class

from pathlib import Path
import json
from datetime import datetime

class AgentTask:
    """Manages the state and dependencies of a long-running, multi-step agent task."""
    
    def __init__(self, task_name: str, state_path: Path = Path('task_state.json')):
        self.task_name = task_name
        self.state_path = state_path
        self.state = self._load_state()

    def _load_state(self):
        """Loads all task states from the JSON file."""
        if not self.state_path.exists():
            return {}
        try:
            with open(self.state_path, 'r') as f:
                return json.load(f)
        except json.JSONDecodeError:
            print(f"Warning: Could not decode {self.state_path}. Starting fresh.")
            return {}

    def _save_state(self):
        """Saves the current task states to the JSON file."""
        with open(self.state_path, 'w') as f:
            json.dump(self.state, f, indent=2)

    def get_task_state(self):
        """Returns the current state of the named task, or initializes a new one."""
        if self.task_name not in self.state:
            self.state[self.task_name] = {
                "status": "PENDING",
                "steps_completed": 0,
                "last_run": None,
                "data": {}
            }
            self._save_state()
        return self.state[self.task_name]

    def update_status(self, new_status: str, data: dict = None):
        """Updates the status and optional data for the current task."""
        task_data = self.get_task_state()
        task_data["status"] = new_status
        task_data["last_run"] = datetime.now().isoformat()
        
        if data:
            task_data["data"].update(data)
            
        self._save_state()
        print(f"Task '{self.task_name}' status updated to: {new_status}")

# Example Usage (for testing the module)
if __name__ == "__main__":
    auditor_task = AgentTask("MFA_SHIPYARD_AUDIT")
    auditor_task.update_status("CONTRACT_CHECKED", {"contract_score": 50})
    
    # Simulate an external dependency being blocked
    state = auditor_task.get_task_state()
    if state["status"] == "CONTRACT_CHECKED":
        # Financial Check would run here
        auditor_task.update_status("COMPLETE", {"final_score": 10, "whale_percent": 18.14})
        
    print(auditor_task.get_task_state())
    
    # Test cleanup
    # Path('task_state.json').unlink() # Uncomment for fresh start
