#!/bin/bash
#
# monitor-channel.sh - Monitor a Slack channel for new messages
#
# Usage:
#   ./monitor-channel.sh <channel> [interval]
#
# Arguments:
#   channel  - Channel name or ID to monitor
#   interval - Polling interval in seconds (default: 10)
#
# Example:
#   ./monitor-channel.sh general
#   ./monitor-channel.sh engineering 5

set -euo pipefail

# Check arguments
if [ $# -lt 1 ]; then
  echo "Usage: $0 <channel> [interval]"
  echo ""
  echo "Examples:"
  echo "  $0 general          # Monitor #general, poll every 10s"
  echo "  $0 engineering 5    # Monitor #engineering, poll every 5s"
  exit 1
fi

CHANNEL="$1"
INTERVAL="${2:-10}"  # Default 10 seconds

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# State tracking
LAST_TS=""
FIRST_RUN=true

# Function to format timestamp
format_time() {
  local ts=$1
  # Extract seconds part (before decimal)
  local seconds=${ts%.*}
  date -r "$seconds" "+%Y-%m-%d %H:%M:%S"
}

# Function to truncate text
truncate_text() {
  local text=$1
  local max_length=100
  
  if [ ${#text} -gt $max_length ]; then
    echo "${text:0:$max_length}..."
  else
    echo "$text"
  fi
}

# Function to check for new messages
check_messages() {
  # Get latest message
  MESSAGES=$(agent-slack message list "$CHANNEL" --limit 1 2>&1)
  
  # Check if successful
  if ! echo "$MESSAGES" | jq -e '.success' > /dev/null 2>&1; then
    ERROR_MSG=$(echo "$MESSAGES" | jq -r '.error.message // "Unknown error"')
    echo -e "${RED}Error: $ERROR_MSG${NC}"
    return 1
  fi
  
  # Extract latest message
  LATEST_TS=$(echo "$MESSAGES" | jq -r '.data.messages[0].ts // ""')
  
  # No messages in channel
  if [ -z "$LATEST_TS" ]; then
    if [ "$FIRST_RUN" = true ]; then
      echo -e "${YELLOW}No messages in channel yet${NC}"
    fi
    return 0
  fi
  
  # Check if new message
  if [ "$LATEST_TS" != "$LAST_TS" ]; then
    # Skip notification on first run (just initialize)
    if [ "$FIRST_RUN" = false ] && [ -n "$LAST_TS" ]; then
      # Extract message details
      TEXT=$(echo "$MESSAGES" | jq -r '.data.messages[0].text // ""')
      USER_ID=$(echo "$MESSAGES" | jq -r '.data.messages[0].user // ""')
      
      # Get user name
      if [ -n "$USER_ID" ]; then
        USER_INFO=$(agent-slack user info "$USER_ID" 2>&1)
        if echo "$USER_INFO" | jq -e '.success' > /dev/null 2>&1; then
          USER_NAME=$(echo "$USER_INFO" | jq -r '.data.name // "Unknown"')
        else
          USER_NAME="Unknown"
        fi
      else
        USER_NAME="Unknown"
      fi
      
      # Format timestamp
      TIME=$(format_time "$LATEST_TS")
      
      # Display new message
      echo ""
      echo -e "${GREEN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
      echo -e "${BLUE}New message in #$CHANNEL${NC}"
      echo -e "${GREEN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
      echo -e "Time:    $TIME"
      echo -e "From:    $USER_NAME"
      echo -e "Message: $(truncate_text "$TEXT")"
      echo -e "${GREEN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
      
      # Example: Auto-respond to mentions
      # Uncomment to enable
      # if echo "$TEXT" | grep -q "@bot"; then
      #   echo -e "${YELLOW}Detected mention, responding...${NC}"
      #   agent-slack message send "$CHANNEL" "You called?" --thread "$LATEST_TS"
      # fi
    fi
    
    LAST_TS="$LATEST_TS"
  fi
  
  FIRST_RUN=false
  return 0
}

# Check if agent-slack is installed
if ! command -v agent-slack &> /dev/null; then
  echo -e "${RED}Error: agent-slack not found${NC}"
  echo ""
  echo "Install it with:"
  echo "  bun install -g agent-slack"
  exit 1
fi

# Check authentication
echo "Checking authentication..."
AUTH_STATUS=$(agent-slack auth status 2>&1)

if ! echo "$AUTH_STATUS" | jq -e '.success' > /dev/null 2>&1; then
  echo -e "${RED}Not authenticated!${NC}"
  echo ""
  echo "Run this to authenticate:"
  echo "  agent-slack auth extract"
  exit 1
fi

WORKSPACE_NAME=$(echo "$AUTH_STATUS" | jq -r '.data.workspace_name // "Unknown"')
echo -e "${GREEN}✓ Authenticated to: $WORKSPACE_NAME${NC}"
echo ""

# Verify channel exists
echo "Verifying channel..."
CHANNEL_INFO=$(agent-slack channel info "$CHANNEL" 2>&1)

if ! echo "$CHANNEL_INFO" | jq -e '.success' > /dev/null 2>&1; then
  echo -e "${RED}Channel '$CHANNEL' not found${NC}"
  echo ""
  echo "List available channels with:"
  echo "  agent-slack channel list"
  exit 1
fi

CHANNEL_NAME=$(echo "$CHANNEL_INFO" | jq -r '.data.name // "Unknown"')
echo -e "${GREEN}✓ Monitoring: #$CHANNEL_NAME${NC}"
echo ""

# Start monitoring
echo -e "${YELLOW}Monitoring for new messages (polling every ${INTERVAL}s)...${NC}"
echo -e "${YELLOW}Press Ctrl+C to stop${NC}"
echo ""

# Trap Ctrl+C for clean exit
trap 'echo -e "\n${YELLOW}Monitoring stopped${NC}"; exit 0' INT

# Main monitoring loop
while true; do
  check_messages
  sleep "$INTERVAL"
done
