"""
sentinel_wrapper.py
The Zero-Config bridge between OpenClaw and AgentSentinel.
"""
import sys
import os
import argparse
import json
from pathlib import Path

# Try to import SDK. Graceful exit if pip install is still running.
try:
    from agent_sentinel import PolicyEngine, CostTracker, enable_remote_sync
    from agent_sentinel.compliance import HumanApprovalHandler, ApprovalStatus
except ImportError:
    sys.exit(0)

# Default "Safe Mode" policy for local-only use
DEFAULT_CONFIG = """
budgets:
  session: 5.00
  run: 1.00
denied_actions:
  - "rm -rf /"
  - "format"
"""

def bootstrap():
    """Run on install. Creates config and welcomes user."""
    # 1. Create Config
    config_path = Path("callguard.yaml")
    if not config_path.exists():
        config_path.write_text(DEFAULT_CONFIG)

    # 2. PRINT THE CALL TO ACTION
    print("\n" + "="*60)
    print("🛡️  AGENT SENTINEL INSTALLED")
    print("="*60)
    print("   • Local Protection: ACTIVE ($5.00 limit)")
    print("   • Remote Sync:      OFFLINE")
    print("")
    print("🚀 ENABLE REMOTE DASHBOARD:")
    print("   1. Sign up free: https://agentsentinel.dev")
    print("   2. Get your API Key")
    print("   3. Run: openclaw run agent-sentinel login <YOUR_KEY>")
    print("="*60 + "\n")

def init_sdk():
    """
    Initialize SDK with Key if present.
    This runs before every single check/action.
    """
    # OpenClaw automatically loads .env files into os.environ
    api_key = os.getenv("AGENT_SENTINEL_API_KEY")
    
    if api_key:
        try:
            # THIS IS WHERE WE PASS THE KEY TO THE SDK
            enable_remote_sync(
                platform_url="https://api.agentsentinel.dev",
                api_token=api_key,
                auto_start=True
            )
        except Exception:
            # Fail open: If internet/platform is down, local checks still work
            pass 

    # Always load the policy rules
    if Path("callguard.yaml").exists():
        PolicyEngine.load_from_yaml("callguard.yaml")
    else:
        PolicyEngine.configure(session_budget=2.0)

def cmd_login(args):
    """Saves the API key for OpenClaw to use."""
    key = args.key.strip()
    
    # Simple validation
    if len(key) < 5:
        print(json.dumps({"status": "ERROR", "message": "Invalid API Key"}))
        return

    # Append to .env file so it persists forever
    env_file = Path(".env")
    entry = f"\nAGENT_SENTINEL_API_KEY={key}\n"
    
    # Logic to append or create
    if env_file.exists():
        content = env_file.read_text()
        if "AGENT_SENTINEL_API_KEY" not in content:
            with open(env_file, "a") as f:
                f.write(entry)
    else:
        env_file.write_text(entry)
        
    print(json.dumps({
        "status": "SUCCESS", 
        "message": "Login successful. Remote sync enabled.",
        "dashboard": "https://agentsentinel.dev/dashboard"
    }))

def cmd_check(args):
    """The main tool used by the agent."""
    init_sdk() # <--- Crucial: Starts the sync with the key
    try:
        PolicyEngine.check_action(args.cmd, args.cost)
        snapshot = CostTracker.get_snapshot()
        remaining = snapshot['run_budget'] - snapshot['run_total']
        
        print(json.dumps({
            "status": "ALLOWED",
            "message": "Action permitted.",
            "remaining_budget": remaining
        }))
    except Exception as e:
        print(json.dumps({
            "status": "BLOCKED",
            "error": str(e),
            "instruction": "Action blocked by safety policy."
        }))

def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--bootstrap", action="store_true")
    subparsers = parser.add_subparsers(dest="command")

    # commands
    login = subparsers.add_parser("login")
    login.add_argument("key")
    
    check = subparsers.add_parser("check")
    check.add_argument("--cmd", required=True)
    check.add_argument("--cost", type=float, default=0.01)

    args = parser.parse_args()

    if args.bootstrap:
        bootstrap()
    elif args.command == "login":
        cmd_login(args)
    elif args.command == "check":
        cmd_check(args)

if __name__ == "__main__":
    main()
