# ERC-8004 Trustless Agents Reference

## Overview

ERC-8004 defines a standard for discovering and establishing trust with AI agents across organizational boundaries. It uses three registries deployed as chain singletons.

## Identity Registry

**Address**: `0x8004A169FB4a3325136EB29fA0ceB6D2e539a432`

### Key Functions

```solidity
// Registration
function register(string agentURI, MetadataEntry[] calldata metadata) external returns (uint256 agentId)
function setAgentURI(uint256 agentId, string calldata newURI) external

// Metadata
function getMetadata(uint256 agentId, string memory metadataKey) external view returns (bytes memory)
function setMetadata(uint256 agentId, string metadataKey, bytes calldata metadataValue) external

// Wallet
function getAgentWallet(uint256 agentId) external view returns (address)
function setAgentWallet(uint256 agentId, address newWallet, uint256 deadline, bytes calldata signature) external
function unsetAgentWallet(uint256 agentId) external

// ERC-721 Standard
function ownerOf(uint256 tokenId) external view returns (address)
function tokenURI(uint256 tokenId) external view returns (string)
function balanceOf(address owner) external view returns (uint256)
```

### Reserved Metadata Keys

- `agentWallet`: Reserved for payment address (cannot be set directly)

## Registration File Schema

```json
{
  "type": "https://eips.ethereum.org/EIPS/eip-8004#registration-v1",
  "name": "string (required)",
  "description": "string (recommended)",
  "image": "url (recommended)",
  "services": [
    {
      "name": "string",
      "endpoint": "url",
      "version": "string (optional)"
    }
  ],
  "x402Support": boolean,
  "active": boolean,
  "registrations": [
    {
      "agentId": "uint256",
      "agentRegistry": "string (eip155:chainId:address)"
    }
  ],
  "supportedTrust": ["reputation", "crypto-economic", "tee-attestation"]
}
```

## Reputation Registry

**Purpose**: Standard interface for feedback signals

### Key Functions

```solidity
function giveFeedback(
    uint256 agentId,
    int128 value,
    uint8 valueDecimals,
    string calldata tag1,
    string calldata tag2,
    string calldata endpoint,
    string calldata feedbackURI,
    bytes32 feedbackHash
) external

function getSummary(
    uint256 agentId,
    address[] calldata clientAddresses,
    string tag1,
    string tag2
) external view returns (uint64 count, int128 summaryValue, uint8 summaryValueDecimals)

function readFeedback(
    uint256 agentId,
    address clientAddress,
    uint64 feedbackIndex
) external view returns (int128 value, uint8 valueDecimals, string tag1, string tag2, bool isRevoked)
```

### Feedback Tags

| Tag | Description | Example |
|-----|-------------|---------|
| starred | Quality rating (0-100) | 87 |
| reachable | Endpoint reachable (binary) | 1 |
| ownerVerified | Endpoint owned by agent owner (binary) | 1 |
| uptime | Endpoint uptime (%) | 9977 (2 decimals) |
| successRate | Endpoint success rate (%) | 89 |
| responseTime | Response time (ms) | 560 |

## Validation Registry

**Purpose**: Request and record validation results from validators

### Key Functions

```solidity
function validationRequest(
    address validatorAddress,
    uint256 agentId,
    string requestURI,
    bytes32 requestHash
) external

function validationResponse(
    bytes32 requestHash,
    uint8 response,
    string responseURI,
    bytes32 responseHash,
    string tag
) external

function getValidationStatus(bytes32 requestHash) external view returns (
    address validatorAddress,
    uint256 agentId,
    uint8 response,
    bytes32 responseHash,
    string tag,
    uint256 lastUpdate
)
```

## Security Considerations

1. **Sybil Attacks**: Reputation can be inflated by fake agents. Use reviewer filtering.

2. **Endpoint Verification**: Agents should verify domain control via `/.well-known/agent-registration.json`

3. **Metadata Trust**: The protocol ensures the registration file corresponds to on-chain agent, but cannot verify capabilities are functional.

4. **On-chain Integrity**: Pointers and hashes cannot be deleted, ensuring audit trail.

## URI Schemes

- `https://` - HTTP endpoints
- `ipfs://` - IPFS content-addressed
- `data:` - Base64-encoded on-chain data

## References

- [ERC-8004 Specification](https://eips.ethereum.org/EIPS/eip-8004)
- [Discussion Forum](https://ethereum-magicians.org/t/erc-8004-trustless-agents/25098)
