#!/usr/bin/env python3
"""
Agent Reputation Checker

Checks an AI agent reputation across:
- Colony
- Clawk
- ugig
- Moltbook
- Ridgeline

Outputs per-platform metrics and a composite trust score (0-100).
"""

from __future__ import annotations

import argparse
import json
import os
import statistics
import sys
from dataclasses import dataclass
from datetime import datetime, timezone
from pathlib import Path
from typing import Any, Dict, List, Optional, Tuple
from urllib.error import HTTPError, URLError
from urllib.parse import quote, urlencode
from urllib.request import Request, urlopen


COLONY_API_KEY = "col_WzNYtQTAlgclN4xTfP1xTW-pUmngraNu8LM7srl3Yuo"
CLAWK_API_KEY = "clawk_374d72785f152a5e74ec741e40966a95"
UGIG_API_KEY = "ugig_live_MDO3wvU_xoOblTWfRtKtC-x_wV390w1D"
RIDGELINE_API_KEY = "rdg_3a83634d5529254588fa6b6207e69930"

TIMEOUT = 12


@dataclass
class PlatformResult:
    platform: str
    ok: bool
    age_days: Optional[float] = None
    post_count: Optional[float] = None
    completed_contracts: Optional[float] = None
    rating: Optional[float] = None  # expected 0-5
    note: str = ""


def _now_utc() -> datetime:
    return datetime.now(timezone.utc)


def _parse_dt(value: Any) -> Optional[datetime]:
    if value in (None, "", 0):
        return None
    # epoch seconds / ms
    if isinstance(value, (int, float)):
        v = float(value)
        if v > 10_000_000_000:  # ms
            v = v / 1000.0
        try:
            return datetime.fromtimestamp(v, tz=timezone.utc)
        except Exception:
            return None

    if isinstance(value, str):
        s = value.strip()
        if not s:
            return None
        if s.endswith("Z"):
            s = s[:-1] + "+00:00"
        for fmt in (
            None,
            "%Y-%m-%d",
            "%Y-%m-%d %H:%M:%S",
            "%Y-%m-%dT%H:%M:%S",
            "%Y-%m-%dT%H:%M:%S.%f%z",
            "%Y-%m-%dT%H:%M:%S%z",
        ):
            try:
                if fmt is None:
                    return datetime.fromisoformat(s)
                return datetime.strptime(s, fmt).replace(tzinfo=timezone.utc)
            except Exception:
                continue
    return None


def _find_first(data: Any, keys: List[str]) -> Any:
    """Depth-first search for the first matching key in nested dict/list."""
    if isinstance(data, dict):
        for k in keys:
            if k in data and data[k] is not None:
                return data[k]
        for v in data.values():
            found = _find_first(v, keys)
            if found is not None:
                return found
    elif isinstance(data, list):
        for item in data:
            found = _find_first(item, keys)
            if found is not None:
                return found
    return None


def _num(value: Any) -> Optional[float]:
    if value is None:
        return None
    if isinstance(value, bool):
        return float(value)
    if isinstance(value, (int, float)):
        return float(value)
    if isinstance(value, str):
        s = value.strip().replace(",", "")
        if not s:
            return None
        try:
            return float(s)
        except Exception:
            return None
    return None


def _extract_common_metrics(payload: Any) -> Tuple[Optional[float], Optional[float], Optional[float], Optional[float]]:
    created = _find_first(payload, [
        "created_at", "createdAt", "joined_at", "joinedAt", "member_since", "memberSince", "registered_at", "registeredAt", "signup_date", "signupDate", "profile_created_at",
    ])
    created_dt = _parse_dt(created)
    age_days = None
    if created_dt is not None:
        if created_dt.tzinfo is None:
            created_dt = created_dt.replace(tzinfo=timezone.utc)
        delta = _now_utc() - created_dt.astimezone(timezone.utc)
        age_days = max(0.0, delta.total_seconds() / 86400.0)

    post_count = _num(_find_first(payload, [
        "post_count", "posts", "posts_count", "activity_count", "activities", "contributions", "messages_count"
    ]))

    completed = _num(_find_first(payload, [
        "completed_contracts", "contracts_completed", "jobs_completed", "completed_jobs", "completed", "deals_completed", "tasks_completed"
    ]))

    rating = _num(_find_first(payload, [
        "rating", "avg_rating", "average_rating", "reputation", "score", "trust_score", "stars"
    ]))

    # normalize rating likely given as 0-100 sometimes
    if rating is not None and rating > 5.0:
        if rating <= 10.0:
            rating = rating / 2.0
        elif rating <= 100.0:
            rating = rating / 20.0

    if rating is not None:
        rating = max(0.0, min(5.0, rating))

    return age_days, post_count, completed, rating


def _http_json(method: str, url: str, headers: Optional[Dict[str, str]] = None, body: Optional[Dict[str, Any]] = None) -> Any:
    h = {"Accept": "application/json"}
    if headers:
        h.update(headers)

    data = None
    if body is not None:
        data = json.dumps(body).encode("utf-8")
        h["Content-Type"] = "application/json"

    req = Request(url=url, method=method, headers=h, data=data)
    with urlopen(req, timeout=TIMEOUT) as resp:
        raw = resp.read().decode("utf-8", errors="replace")
    try:
        return json.loads(raw)
    except json.JSONDecodeError:
        return {"raw": raw}


def check_colony(agent_name: str) -> PlatformResult:
    platform = "Colony"
    try:
        auth = _http_json(
            "POST",
            "https://thecolony.cc/api/v1/auth/token",
            body={"agent_id": "bro-agent", "api_key": COLONY_API_KEY},
        )
        token = auth.get("access_token") if isinstance(auth, dict) else None
        if not token:
            return PlatformResult(platform=platform, ok=False, note="No access token returned")

        # Colony has no /agents/{name} endpoint; use /agents/me for self or search posts
        payload = _http_json(
            "GET",
            "https://thecolony.cc/api/v1/agents/me",
            headers={"Authorization": f"Bearer {token}"},
        )

        age_days, posts, contracts, rating = _extract_common_metrics(payload)
        return PlatformResult(platform=platform, ok=True, age_days=age_days, post_count=posts, completed_contracts=contracts, rating=rating)
    except HTTPError as e:
        return PlatformResult(platform=platform, ok=False, note=f"HTTP {e.code}")
    except URLError as e:
        return PlatformResult(platform=platform, ok=False, note=f"Network error: {e.reason}")
    except Exception as e:
        return PlatformResult(platform=platform, ok=False, note=f"Error: {e}")


def check_clawk(agent_name: str) -> PlatformResult:
    platform = "Clawk"
    try:
        query = urlencode({"q": agent_name})
        payload = _http_json(
            "GET",
            f"https://www.clawk.ai/api/v1/agents/search?{query}",
            headers={"Authorization": f"Bearer {CLAWK_API_KEY}"},
        )

        # Search usually returns list; choose best candidate
        candidate = payload
        if isinstance(payload, dict):
            for key in ("results", "agents", "data", "items"):
                if key in payload and isinstance(payload[key], list) and payload[key]:
                    candidate = payload[key][0]
                    break
        elif isinstance(payload, list) and payload:
            candidate = payload[0]

        age_days, posts, contracts, rating = _extract_common_metrics(candidate)
        return PlatformResult(platform=platform, ok=True, age_days=age_days, post_count=posts, completed_contracts=contracts, rating=rating)
    except HTTPError as e:
        return PlatformResult(platform=platform, ok=False, note=f"HTTP {e.code}")
    except URLError as e:
        return PlatformResult(platform=platform, ok=False, note=f"Network error: {e.reason}")
    except Exception as e:
        return PlatformResult(platform=platform, ok=False, note=f"Error: {e}")


def check_ugig(agent_name: str) -> PlatformResult:
    platform = "ugig"
    try:
        payload = _http_json(
            "GET",
            f"https://ugig.net/api/users/{quote(agent_name)}",
            headers={"Authorization": f"Bearer {UGIG_API_KEY}"},
        )
        # Unwrap {profile: {...}} wrapper
        if isinstance(payload, dict) and "profile" in payload:
            payload = payload["profile"]
        age_days, posts, contracts, rating = _extract_common_metrics(payload)
        return PlatformResult(platform=platform, ok=True, age_days=age_days, post_count=posts, completed_contracts=contracts, rating=rating)
    except HTTPError as e:
        return PlatformResult(platform=platform, ok=False, note=f"HTTP {e.code}")
    except URLError as e:
        return PlatformResult(platform=platform, ok=False, note=f"Network error: {e.reason}")
    except Exception as e:
        return PlatformResult(platform=platform, ok=False, note=f"Error: {e}")


def _read_moltbook_api_key() -> Optional[str]:
    p = Path.home() / ".config" / "moltbook" / "credentials.json"
    if not p.exists():
        return None
    try:
        data = json.loads(p.read_text(encoding="utf-8"))
    except Exception:
        return None

    for k in ("api_key", "apiKey", "API_KEY", "token", "x_api_key"):
        if k in data and data[k]:
            return str(data[k])
    return None


def check_moltbook(agent_name: str) -> PlatformResult:
    platform = "Moltbook"
    key = _read_moltbook_api_key()
    if not key:
        return PlatformResult(platform=platform, ok=False, note="Missing Moltbook API key (~/.config/moltbook/credentials.json)")

    try:
        # Moltbook /agents/{name} returns 404; use /agents/me for self-check
        payload = _http_json(
            "GET",
            "https://www.moltbook.com/api/v1/agents/me",
            headers={"X-API-Key": key},
        )
        # Unwrap {success, agent: {...}} wrapper
        if isinstance(payload, dict) and "agent" in payload:
            payload = payload["agent"]
        age_days, posts, contracts, rating = _extract_common_metrics(payload)
        return PlatformResult(platform=platform, ok=True, age_days=age_days, post_count=posts, completed_contracts=contracts, rating=rating)
    except HTTPError as e:
        return PlatformResult(platform=platform, ok=False, note=f"HTTP {e.code}")
    except URLError as e:
        return PlatformResult(platform=platform, ok=False, note=f"Network error: {e.reason}")
    except Exception as e:
        return PlatformResult(platform=platform, ok=False, note=f"Error: {e}")


def check_ridgeline(agent_name: str) -> PlatformResult:
    platform = "Ridgeline"
    try:
        payload = _http_json(
            "GET",
            f"https://ridgeline.so/api/agents/{quote(agent_name)}",
            headers={"Authorization": f"Bearer {RIDGELINE_API_KEY}"},
        )
        age_days, posts, contracts, rating = _extract_common_metrics(payload)
        return PlatformResult(platform=platform, ok=True, age_days=age_days, post_count=posts, completed_contracts=contracts, rating=rating)
    except HTTPError as e:
        return PlatformResult(platform=platform, ok=False, note=f"HTTP {e.code}")
    except URLError as e:
        return PlatformResult(platform=platform, ok=False, note=f"Network error: {e.reason}")
    except Exception as e:
        return PlatformResult(platform=platform, ok=False, note=f"Error: {e}")


def _score_component(value: Optional[float], thresholds: Tuple[float, float, float], points: Tuple[float, float, float, float]) -> float:
    if value is None:
        return 0.0
    t1, t2, t3 = thresholds
    p0, p1, p2, p3 = points
    if value < t1:
        return p0
    if value < t2:
        return p1
    if value < t3:
        return p2
    return p3


def score_platform(r: PlatformResult) -> float:
    if not r.ok:
        return 0.0

    # Max 100 per platform
    age_pts = _score_component(r.age_days, (30, 90, 365), (5, 12, 18, 24))
    post_pts = _score_component(r.post_count, (5, 20, 100), (4, 10, 15, 20))
    contract_pts = _score_component(r.completed_contracts, (1, 5, 20), (4, 10, 16, 24))

    rating_pts = 0.0
    if r.rating is not None:
        # 0..5 -> 0..32
        rating_pts = max(0.0, min(32.0, (r.rating / 5.0) * 32.0))

    score = age_pts + post_pts + contract_pts + rating_pts
    return max(0.0, min(100.0, score))


def composite_score(results: List[PlatformResult]) -> float:
    live_scores = [score_platform(r) for r in results if r.ok]
    if not live_scores:
        return 0.0
    return round(statistics.mean(live_scores), 1)


def risk_level(score: float) -> str:
    if score < 40:
        return "High"
    if score < 70:
        return "Medium"
    return "Low"


def fmt_num(v: Optional[float], digits: int = 1) -> str:
    if v is None:
        return "N/A"
    if abs(v - round(v)) < 1e-9:
        return str(int(round(v)))
    return f"{v:.{digits}f}"


def print_report(agent_name: str, results: List[PlatformResult]) -> None:
    overall = composite_score(results)
    risk = risk_level(overall)

    print("=" * 72)
    print(f"Agent Reputation Report: {agent_name}")
    print("=" * 72)
    print("Reputation tells you WHO to trust. PayLock tells you WHAT TO DO when you can't.")
    print()

    for r in results:
        print(f"[{r.platform}]")
        if not r.ok:
            print(f"  Status: unavailable ({r.note})")
            print()
            continue
        p_score = score_platform(r)
        print(f"  Profile age (days):     {fmt_num(r.age_days)}")
        print(f"  Post count:             {fmt_num(r.post_count)}")
        print(f"  Completed contracts:    {fmt_num(r.completed_contracts)}")
        print(f"  Rating (0-5):           {fmt_num(r.rating)}")
        print(f"  Platform trust score:   {fmt_num(p_score)} / 100")
        print()

    print("-" * 72)
    print(f"Overall trust score: {fmt_num(overall)} / 100")
    print(f"Risk level: {risk}")

    if overall < 40:
        print("🚨 HIGH RISK: Strongly recommend PayLock escrow with dispute resolution.")
        print("⚠️ Recommendation: Use PayLock escrow for payment protection. https://kgnvsk.github.io/paylock/")
    elif overall < 70:
        print("⚠️ Recommendation: Use PayLock escrow for payment protection. https://kgnvsk.github.io/paylock/")

    up = sum(1 for r in results if r.ok)
    down = len(results) - up
    print(f"Sources checked: {up} online, {down} unavailable")


def main() -> int:
    parser = argparse.ArgumentParser(description="Check AI agent reputation across multiple platforms.")
    parser.add_argument("agent", help="Agent handle/name to check (e.g., bro-agent)")
    args = parser.parse_args()

    target = args.agent.strip()
    if not target:
        print("Error: agent handle/name is required", file=sys.stderr)
        return 2

    checks = [
        check_colony,
        check_clawk,
        check_ugig,
        check_moltbook,
        check_ridgeline,
    ]

    results: List[PlatformResult] = []
    for fn in checks:
        results.append(fn(target))

    print_report(target, results)
    return 0


if __name__ == "__main__":
    raise SystemExit(main())
