#!/usr/bin/env node
import { parseUnits } from 'viem';
import { program } from 'commander';
import { 
  loadKey, 
  getPublicClient, 
  getWalletClient, 
  CONTRACTS, 
  REGISTRY_ABI, 
  ERC20_ABI,
  output, 
  DEFAULT_CHAIN 
} from './config.js';

program
  .name('vouch')
  .description('Vouch for another agent by staking USDC')
  .requiredOption('--identity <hash>', 'Identity hash to vouch for')
  .requiredOption('--amount <usdc>', 'USDC amount to stake as vouch')
  .option('--chain <name>', 'Chain to use', DEFAULT_CHAIN)
  .option('--json', 'Output as JSON')
  .option('--yes', 'Skip confirmation')
  .parse();

const opts = program.opts();

async function main() {
  const keyData = loadKey();
  if (!keyData) {
    throw new Error('No keypair found. Run setup.js first.');
  }

  const publicClient = getPublicClient(opts.chain);
  const walletClient = getWalletClient(opts.chain);
  const contracts = CONTRACTS[opts.chain];

  if (contracts.registry === '0x0000000000000000000000000000000000000000') {
    throw new Error(`Registry not deployed on ${opts.chain}`);
  }

  const vouchAmount = parseUnits(opts.amount, 6);

  // Get target identity info
  const identity = await publicClient.readContract({
    address: contracts.registry,
    abi: REGISTRY_ABI,
    functionName: 'getIdentity',
    args: [opts.identity],
  });

  const targetName = identity[2];
  const currentVouches = Number(identity[7]) / 1e6;

  if (!opts.json && !opts.yes) {
    console.log('📝 Vouch Details:');
    console.log(`  Target: ${targetName} (${opts.identity.slice(0, 10)}...)`);
    console.log(`  Amount: ${opts.amount} USDC`);
    console.log(`  Current vouches: ${currentVouches.toFixed(2)} USDC`);
    console.log('');
    console.log('⚠️  Your stake is at risk if this agent acts maliciously.');
    console.log('');
    console.log('Run with --yes to confirm, or Ctrl+C to cancel.');
    process.exit(0);
  }

  // Check and approve USDC
  if (!opts.json) console.log('Checking USDC allowance...');

  const allowance = await publicClient.readContract({
    address: contracts.usdc,
    abi: ERC20_ABI,
    functionName: 'allowance',
    args: [keyData.address, contracts.registry],
  });

  if (allowance < vouchAmount) {
    if (!opts.json) console.log(`Approving ${opts.amount} USDC...`);
    
    const approveTx = await walletClient.writeContract({
      address: contracts.usdc,
      abi: ERC20_ABI,
      functionName: 'approve',
      args: [contracts.registry, vouchAmount],
    });

    await publicClient.waitForTransactionReceipt({ hash: approveTx });
    if (!opts.json) console.log('Approval confirmed!');
  }

  // Submit vouch
  if (!opts.json) console.log(`Vouching for ${targetName}...`);

  const tx = await walletClient.writeContract({
    address: contracts.registry,
    abi: REGISTRY_ABI,
    functionName: 'vouch',
    args: [opts.identity, vouchAmount],
  });

  await publicClient.waitForTransactionReceipt({ hash: tx });

  // Get updated vouch total
  const updatedIdentity = await publicClient.readContract({
    address: contracts.registry,
    abi: REGISTRY_ABI,
    functionName: 'getIdentity',
    args: [opts.identity],
  });

  const newVouches = Number(updatedIdentity[7]) / 1e6;

  const result = {
    success: true,
    target: {
      name: targetName,
      identityHash: opts.identity,
    },
    vouchAmount: opts.amount,
    totalVouches: newVouches.toFixed(2),
    txHash: tx,
    voucher: keyData.address,
  };

  if (opts.json) {
    output(result, true);
  } else {
    console.log('');
    console.log('✅ Vouch submitted!');
    console.log('');
    console.log(`You vouched ${opts.amount} USDC for ${targetName}`);
    console.log(`Total vouches: ${newVouches.toFixed(2)} USDC`);
  }
}

main().catch((err) => {
  if (opts.json) {
    output({ success: false, error: err.message }, true);
  } else {
    console.error('Error:', err.message);
  }
  process.exit(1);
});
