#!/usr/bin/env node
import { verifyMessage, hashMessage } from 'viem';
import { program } from 'commander';
import { getPublicClient, CONTRACTS, REGISTRY_ABI, output, DEFAULT_CHAIN } from './config.js';

program
  .name('verify')
  .description('Verify a signature from an agent')
  .requiredOption('--message <text>', 'Original message that was signed')
  .requiredOption('--signature <hex>', 'Signature to verify')
  .option('--signer <address>', 'Signer address (for off-chain verification)')
  .option('--identity <hash>', 'Identity hash (for on-chain verification)')
  .option('--chain <name>', 'Chain to use', DEFAULT_CHAIN)
  .option('--json', 'Output as JSON')
  .parse();

const opts = program.opts();

async function main() {
  if (!opts.signer && !opts.identity) {
    throw new Error('Must provide either --signer or --identity');
  }

  let valid = false;
  let recoveredAddress = null;
  let onChainIdentity = null;

  // Off-chain verification (just check signature)
  if (opts.signer) {
    valid = await verifyMessage({
      address: opts.signer,
      message: opts.message,
      signature: opts.signature,
    });
    recoveredAddress = opts.signer;
  }

  // On-chain verification (check against registry)
  if (opts.identity) {
    const client = getPublicClient(opts.chain);
    const contracts = CONTRACTS[opts.chain];
    
    if (contracts.registry === '0x0000000000000000000000000000000000000000') {
      throw new Error(`Registry not deployed on ${opts.chain}. Use off-chain verification with --signer.`);
    }

    const messageHash = hashMessage(opts.message);
    
    // Verify on-chain
    valid = await client.readContract({
      address: contracts.registry,
      abi: REGISTRY_ABI,
      functionName: 'verifySignature',
      args: [opts.identity, messageHash, opts.signature],
    });

    // Get identity details
    const identity = await client.readContract({
      address: contracts.registry,
      abi: REGISTRY_ABI,
      functionName: 'getIdentity',
      args: [opts.identity],
    });

    const platforms = await client.readContract({
      address: contracts.registry,
      abi: REGISTRY_ABI,
      functionName: 'getLinkedPlatforms',
      args: [opts.identity],
    });

    onChainIdentity = {
      owner: identity[0],
      signingKey: identity[1],
      name: identity[2],
      metadataUri: identity[3],
      stakedAmount: identity[4].toString(),
      registeredAt: new Date(Number(identity[5]) * 1000).toISOString(),
      platforms: platforms,
    };
    
    recoveredAddress = identity[1]; // signingKey
  }

  const result = {
    success: true,
    valid,
    message: opts.message,
    signature: opts.signature,
    signer: recoveredAddress,
    identity: onChainIdentity,
    verifiedAt: new Date().toISOString(),
  };

  if (opts.json) {
    output(result, true);
  } else {
    if (valid) {
      console.log('✅ Signature is VALID');
    } else {
      console.log('❌ Signature is INVALID');
    }
    console.log('');
    console.log(`Message: ${opts.message}`);
    console.log(`Signer: ${recoveredAddress}`);
    
    if (onChainIdentity) {
      console.log('');
      console.log('On-chain identity:');
      console.log(`  Name: ${onChainIdentity.name}`);
      console.log(`  Stake: ${(Number(onChainIdentity.stakedAmount) / 1e6).toFixed(2)} USDC`);
      console.log(`  Registered: ${onChainIdentity.registeredAt}`);
      if (onChainIdentity.platforms.length > 0) {
        console.log(`  Platforms: ${onChainIdentity.platforms.join(', ')}`);
      }
    }
  }
}

main().catch((err) => {
  if (opts.json) {
    output({ success: false, error: err.message }, true);
  } else {
    console.error('Error:', err.message);
  }
  process.exit(1);
});
