#!/usr/bin/env node
import { privateKeyToAccount } from 'viem/accounts';
import { hashMessage } from 'viem';
import { program } from 'commander';
import { loadKey, output } from './config.js';

program
  .name('sign')
  .description('Sign a message with your identity key')
  .requiredOption('--message <text>', 'Message to sign')
  .option('--json', 'Output as JSON')
  .parse();

const opts = program.opts();

async function main() {
  const keyData = loadKey();
  if (!keyData) {
    throw new Error('No keypair found. Run setup.js first.');
  }

  const account = privateKeyToAccount(keyData.privateKey);
  
  // Sign the message
  const signature = await account.signMessage({
    message: opts.message,
  });

  // Get message hash for on-chain verification
  const messageHash = hashMessage(opts.message);

  const result = {
    success: true,
    message: opts.message,
    messageHash,
    signature,
    signerAddress: account.address,
    identityHash: keyData.identityHash || null,
    timestamp: new Date().toISOString(),
  };

  if (opts.json) {
    output(result, true);
  } else {
    console.log('✅ Message signed!');
    console.log('');
    console.log(`Message: ${opts.message}`);
    console.log(`Signer: ${account.address}`);
    console.log(`Signature: ${signature}`);
    if (keyData.identityHash) {
      console.log(`Identity: ${keyData.identityHash}`);
    }
    console.log('');
    console.log('To verify this signature:');
    console.log(`node scripts/verify.js --message "${opts.message}" --signature "${signature}" --signer "${account.address}"`);
  }
}

main().catch((err) => {
  if (opts.json) {
    output({ success: false, error: err.message }, true);
  } else {
    console.error('Error:', err.message);
  }
  process.exit(1);
});
