#!/usr/bin/env node
import { generatePrivateKey, privateKeyToAccount } from 'viem/accounts';
import { program } from 'commander';
import { loadKey, saveKey, output, CONFIG_DIR, KEY_FILE } from './config.js';
import fs from 'fs';

program
  .name('setup')
  .description('Set up your agent identity keypair')
  .option('--json', 'Output as JSON')
  .option('--force', 'Overwrite existing keypair')
  .option('--import <privateKey>', 'Import existing private key')
  .parse();

const opts = program.opts();

async function main() {
  // Check for existing key
  const existingKey = loadKey();
  if (existingKey && !opts.force) {
    if (opts.json) {
      output({
        success: false,
        error: 'Keypair already exists. Use --force to overwrite.',
        address: existingKey.address,
      }, true);
    } else {
      console.error(`Keypair already exists at ${KEY_FILE}`);
      console.error(`Address: ${existingKey.address}`);
      console.error('Use --force to overwrite.');
    }
    process.exit(1);
  }

  let privateKey;
  
  if (opts.import) {
    // Import existing key
    privateKey = opts.import.startsWith('0x') ? opts.import : `0x${opts.import}`;
  } else {
    // Generate new keypair
    privateKey = generatePrivateKey();
  }

  const account = privateKeyToAccount(privateKey);
  
  const keyData = {
    address: account.address,
    privateKey: privateKey,
    createdAt: new Date().toISOString(),
    identityHash: null, // Set after registration
  };

  saveKey(keyData);

  if (opts.json) {
    output({
      success: true,
      address: account.address,
      keyFile: KEY_FILE,
      message: opts.import ? 'Imported existing key' : 'Generated new keypair',
    }, true);
  } else {
    console.log('✅ Agent identity keypair created!');
    console.log(`📍 Address: ${account.address}`);
    console.log(`📁 Key file: ${KEY_FILE}`);
    console.log('');
    console.log('Next steps:');
    console.log('1. Fund this address with Base Sepolia ETH and USDC');
    console.log('2. Run: node scripts/register.js --name "your-agent-name"');
  }
}

main().catch((err) => {
  if (opts.json) {
    output({ success: false, error: err.message }, true);
  } else {
    console.error('Error:', err.message);
  }
  process.exit(1);
});
