#!/usr/bin/env node
import { parseUnits } from 'viem';
import { program } from 'commander';
import { 
  loadKey, 
  saveKey,
  getPublicClient, 
  getWalletClient, 
  CONTRACTS, 
  REGISTRY_ABI, 
  ERC20_ABI,
  output, 
  DEFAULT_CHAIN 
} from './config.js';

program
  .name('register')
  .description('Register your agent identity on-chain')
  .requiredOption('--name <name>', 'Your agent name')
  .option('--metadata <uri>', 'IPFS URI to extended metadata', '')
  .option('--stake <amount>', 'USDC amount to stake', '1.0')
  .option('--chain <name>', 'Chain to use', DEFAULT_CHAIN)
  .option('--json', 'Output as JSON')
  .option('--yes', 'Skip confirmation')
  .parse();

const opts = program.opts();

async function main() {
  const keyData = loadKey();
  if (!keyData) {
    throw new Error('No keypair found. Run setup.js first.');
  }

  if (keyData.identityHash) {
    throw new Error(`Already registered with identity: ${keyData.identityHash}`);
  }

  const publicClient = getPublicClient(opts.chain);
  const walletClient = getWalletClient(opts.chain);
  const contracts = CONTRACTS[opts.chain];

  if (contracts.registry === '0x0000000000000000000000000000000000000000') {
    throw new Error(`Registry not deployed on ${opts.chain}`);
  }

  const stakeAmount = parseUnits(opts.stake, 6); // USDC has 6 decimals
  const signingKey = keyData.address;

  if (!opts.json && !opts.yes) {
    console.log('📝 Registration Details:');
    console.log(`  Name: ${opts.name}`);
    console.log(`  Signing Key: ${signingKey}`);
    console.log(`  Stake: ${opts.stake} USDC`);
    console.log(`  Chain: ${opts.chain}`);
    console.log('');
    console.log('This will:');
    console.log(`  1. Approve ${opts.stake} USDC for the registry`);
    console.log('  2. Register your identity on-chain');
    console.log('');
    console.log('Run with --yes to confirm, or Ctrl+C to cancel.');
    process.exit(0);
  }

  // Step 1: Check USDC allowance and approve if needed
  if (!opts.json) console.log('Checking USDC allowance...');
  
  const allowance = await publicClient.readContract({
    address: contracts.usdc,
    abi: ERC20_ABI,
    functionName: 'allowance',
    args: [keyData.address, contracts.registry],
  });

  if (allowance < stakeAmount) {
    if (!opts.json) console.log(`Approving ${opts.stake} USDC...`);
    
    const approveTx = await walletClient.writeContract({
      address: contracts.usdc,
      abi: ERC20_ABI,
      functionName: 'approve',
      args: [contracts.registry, stakeAmount],
    });

    if (!opts.json) console.log(`Approval tx: ${approveTx}`);
    
    // Wait for approval
    await publicClient.waitForTransactionReceipt({ hash: approveTx });
    if (!opts.json) console.log('Approval confirmed!');
  }

  // Step 2: Register identity
  if (!opts.json) console.log('Registering identity...');

  const registerTx = await walletClient.writeContract({
    address: contracts.registry,
    abi: REGISTRY_ABI,
    functionName: 'register',
    args: [opts.name, opts.metadata, signingKey, stakeAmount],
  });

  if (!opts.json) console.log(`Registration tx: ${registerTx}`);

  // Wait for registration and get receipt
  const receipt = await publicClient.waitForTransactionReceipt({ hash: registerTx });

  // Get identity hash from ownerToIdentity
  const identityHash = await publicClient.readContract({
    address: contracts.registry,
    abi: REGISTRY_ABI,
    functionName: 'ownerToIdentity',
    args: [keyData.address],
  });

  // Update local key file with identity hash
  keyData.identityHash = identityHash;
  keyData.registeredAt = new Date().toISOString();
  keyData.name = opts.name;
  saveKey(keyData);

  const result = {
    success: true,
    name: opts.name,
    identityHash,
    owner: keyData.address,
    signingKey,
    stake: opts.stake,
    txHash: registerTx,
    chain: opts.chain,
  };

  if (opts.json) {
    output(result, true);
  } else {
    console.log('');
    console.log('✅ Identity registered!');
    console.log('');
    console.log(`Name: ${opts.name}`);
    console.log(`Identity Hash: ${identityHash}`);
    console.log(`Stake: ${opts.stake} USDC`);
    console.log('');
    console.log('Next steps:');
    console.log('• Link platforms: node scripts/link.js --platform "moltbook:your-username"');
    console.log('• Sign messages: node scripts/sign.js --message "Hello world"');
  }
}

main().catch((err) => {
  if (opts.json) {
    output({ success: false, error: err.message }, true);
  } else {
    console.error('Error:', err.message);
  }
  process.exit(1);
});
