#!/usr/bin/env node
import { program } from 'commander';
import { getPublicClient, CONTRACTS, REGISTRY_ABI, output, DEFAULT_CHAIN } from './config.js';

program
  .name('lookup')
  .description('Look up an agent identity')
  .option('--identity <hash>', 'Identity hash to look up')
  .option('--address <address>', 'Look up by owner address')
  .option('--chain <name>', 'Chain to use', DEFAULT_CHAIN)
  .option('--json', 'Output as JSON')
  .parse();

const opts = program.opts();

async function main() {
  if (!opts.identity && !opts.address) {
    throw new Error('Must provide either --identity or --address');
  }

  const client = getPublicClient(opts.chain);
  const contracts = CONTRACTS[opts.chain];

  if (contracts.registry === '0x0000000000000000000000000000000000000000') {
    throw new Error(`Registry not deployed on ${opts.chain}`);
  }

  let identityHash = opts.identity;

  // If looking up by address, get identity hash first
  if (opts.address) {
    identityHash = await client.readContract({
      address: contracts.registry,
      abi: REGISTRY_ABI,
      functionName: 'ownerToIdentity',
      args: [opts.address],
    });

    if (identityHash === '0x0000000000000000000000000000000000000000000000000000000000000000') {
      throw new Error(`No identity found for address ${opts.address}`);
    }
  }

  // Get identity details
  const identity = await client.readContract({
    address: contracts.registry,
    abi: REGISTRY_ABI,
    functionName: 'getIdentity',
    args: [identityHash],
  });

  // Get linked platforms
  const platforms = await client.readContract({
    address: contracts.registry,
    abi: REGISTRY_ABI,
    functionName: 'getLinkedPlatforms',
    args: [identityHash],
  });

  // Check if active
  const active = await client.readContract({
    address: contracts.registry,
    abi: REGISTRY_ABI,
    functionName: 'isActive',
    args: [identityHash],
  });

  const result = {
    success: true,
    identityHash,
    owner: identity[0],
    signingKey: identity[1],
    name: identity[2],
    metadataUri: identity[3],
    stakedAmount: (Number(identity[4]) / 1e6).toFixed(2),
    registeredAt: new Date(Number(identity[5]) * 1000).toISOString(),
    deactivatedAt: identity[6] > 0 ? new Date(Number(identity[6]) * 1000).toISOString() : null,
    totalVouches: (Number(identity[7]) / 1e6).toFixed(2),
    platforms,
    active,
  };

  if (opts.json) {
    output(result, true);
  } else {
    console.log(`🤖 Agent Identity: ${result.name}`);
    console.log('');
    console.log(`Identity Hash: ${identityHash}`);
    console.log(`Owner: ${result.owner}`);
    console.log(`Signing Key: ${result.signingKey}`);
    console.log(`Status: ${active ? '✅ Active' : '❌ Inactive'}`);
    console.log('');
    console.log(`Stake: ${result.stakedAmount} USDC`);
    console.log(`Vouches: ${result.totalVouches} USDC`);
    console.log(`Registered: ${result.registeredAt}`);
    
    if (platforms.length > 0) {
      console.log('');
      console.log('Linked Platforms:');
      platforms.forEach(p => console.log(`  • ${p}`));
    }
    
    if (result.metadataUri) {
      console.log('');
      console.log(`Metadata: ${result.metadataUri}`);
    }
  }
}

main().catch((err) => {
  if (opts.json) {
    output({ success: false, error: err.message }, true);
  } else {
    console.error('Error:', err.message);
  }
  process.exit(1);
});
