#!/usr/bin/env node
import { program } from 'commander';
import { 
  loadKey, 
  getPublicClient, 
  getWalletClient, 
  CONTRACTS, 
  REGISTRY_ABI, 
  output, 
  DEFAULT_CHAIN 
} from './config.js';

program
  .name('link')
  .description('Link a platform account to your identity')
  .requiredOption('--platform <string>', 'Platform identifier (e.g., "moltbook:g1itchbot")')
  .option('--chain <name>', 'Chain to use', DEFAULT_CHAIN)
  .option('--json', 'Output as JSON')
  .parse();

const opts = program.opts();

async function main() {
  const keyData = loadKey();
  if (!keyData) {
    throw new Error('No keypair found. Run setup.js first.');
  }

  if (!keyData.identityHash) {
    throw new Error('Not registered yet. Run register.js first.');
  }

  // Validate platform format
  if (!opts.platform.includes(':')) {
    throw new Error('Platform must be in format "platform:username" (e.g., "moltbook:g1itchbot")');
  }

  const publicClient = getPublicClient(opts.chain);
  const walletClient = getWalletClient(opts.chain);
  const contracts = CONTRACTS[opts.chain];

  if (contracts.registry === '0x0000000000000000000000000000000000000000') {
    throw new Error(`Registry not deployed on ${opts.chain}`);
  }

  if (!opts.json) console.log(`Linking platform: ${opts.platform}...`);

  const tx = await walletClient.writeContract({
    address: contracts.registry,
    abi: REGISTRY_ABI,
    functionName: 'linkPlatform',
    args: [opts.platform],
  });

  if (!opts.json) console.log(`Transaction: ${tx}`);

  await publicClient.waitForTransactionReceipt({ hash: tx });

  // Get updated platform list
  const platforms = await publicClient.readContract({
    address: contracts.registry,
    abi: REGISTRY_ABI,
    functionName: 'getLinkedPlatforms',
    args: [keyData.identityHash],
  });

  const result = {
    success: true,
    platform: opts.platform,
    identityHash: keyData.identityHash,
    allPlatforms: platforms,
    txHash: tx,
  };

  if (opts.json) {
    output(result, true);
  } else {
    console.log('');
    console.log('✅ Platform linked!');
    console.log('');
    console.log(`Identity: ${keyData.identityHash}`);
    console.log('Linked platforms:');
    platforms.forEach(p => console.log(`  • ${p}`));
  }
}

main().catch((err) => {
  if (opts.json) {
    output({ success: false, error: err.message }, true);
  } else {
    console.error('Error:', err.message);
  }
  process.exit(1);
});
