import { createPublicClient, createWalletClient, http } from 'viem';
import { baseSepolia, base } from 'viem/chains';
import { privateKeyToAccount } from 'viem/accounts';
import fs from 'fs';
import path from 'path';
import os from 'os';

// Config paths
export const CONFIG_DIR = path.join(os.homedir(), '.agent-identity');
export const KEY_FILE = path.join(CONFIG_DIR, 'key.json');

// Contract addresses (update after deployment)
export const CONTRACTS = {
  baseSepolia: {
    registry: process.env.REGISTRY_ADDRESS || '0x818353E08861C6b5EA1545743862F6211f01a6E0',
    usdc: '0x036CbD53842c5426634e7929541eC2318f3dCF7e',
  },
  base: {
    registry: process.env.REGISTRY_ADDRESS || '0x0000000000000000000000000000000000000000',
    usdc: '0x833589fCD6eDb6E08f4c7C32D4f71b54bdA02913',
  },
};

// Default chain
export const DEFAULT_CHAIN = 'baseSepolia';

// Get chain config
export function getChain(chainName = DEFAULT_CHAIN) {
  const chains = {
    baseSepolia,
    base,
  };
  return chains[chainName] || baseSepolia;
}

// Load keypair from file
export function loadKey() {
  if (!fs.existsSync(KEY_FILE)) {
    return null;
  }
  const data = JSON.parse(fs.readFileSync(KEY_FILE, 'utf8'));
  return data;
}

// Save keypair to file
export function saveKey(keyData) {
  if (!fs.existsSync(CONFIG_DIR)) {
    fs.mkdirSync(CONFIG_DIR, { recursive: true, mode: 0o700 });
  }
  fs.writeFileSync(KEY_FILE, JSON.stringify(keyData, null, 2), { mode: 0o600 });
}

// Get public client
export function getPublicClient(chainName = DEFAULT_CHAIN) {
  const chain = getChain(chainName);
  return createPublicClient({
    chain,
    transport: http(),
  });
}

// Get wallet client
export function getWalletClient(chainName = DEFAULT_CHAIN) {
  const keyData = loadKey();
  if (!keyData) {
    throw new Error('No keypair found. Run setup.js first.');
  }
  
  const chain = getChain(chainName);
  const account = privateKeyToAccount(keyData.privateKey);
  
  return createWalletClient({
    account,
    chain,
    transport: http(),
  });
}

// Registry contract ABI (simplified for now)
export const REGISTRY_ABI = [
  {
    name: 'register',
    type: 'function',
    inputs: [
      { name: 'name', type: 'string' },
      { name: 'metadataUri', type: 'string' },
      { name: 'signingKey', type: 'address' },
      { name: 'stakeAmount', type: 'uint256' },
    ],
    outputs: [{ name: 'identityHash', type: 'bytes32' }],
  },
  {
    name: 'linkPlatform',
    type: 'function',
    inputs: [{ name: 'platform', type: 'string' }],
    outputs: [],
  },
  {
    name: 'vouch',
    type: 'function',
    inputs: [
      { name: 'identityHash', type: 'bytes32' },
      { name: 'amount', type: 'uint256' },
    ],
    outputs: [],
  },
  {
    name: 'verifySignature',
    type: 'function',
    inputs: [
      { name: 'identityHash', type: 'bytes32' },
      { name: 'messageHash', type: 'bytes32' },
      { name: 'signature', type: 'bytes' },
    ],
    outputs: [{ name: '', type: 'bool' }],
  },
  {
    name: 'getIdentity',
    type: 'function',
    inputs: [{ name: 'identityHash', type: 'bytes32' }],
    outputs: [
      { name: 'owner', type: 'address' },
      { name: 'signingKey', type: 'address' },
      { name: 'name', type: 'string' },
      { name: 'metadataUri', type: 'string' },
      { name: 'stakedAmount', type: 'uint256' },
      { name: 'registeredAt', type: 'uint256' },
      { name: 'deactivatedAt', type: 'uint256' },
      { name: 'totalVouchesReceived', type: 'uint256' },
    ],
  },
  {
    name: 'getLinkedPlatforms',
    type: 'function',
    inputs: [{ name: 'identityHash', type: 'bytes32' }],
    outputs: [{ name: '', type: 'string[]' }],
  },
  {
    name: 'ownerToIdentity',
    type: 'function',
    inputs: [{ name: 'owner', type: 'address' }],
    outputs: [{ name: '', type: 'bytes32' }],
  },
  {
    name: 'isActive',
    type: 'function',
    inputs: [{ name: 'identityHash', type: 'bytes32' }],
    outputs: [{ name: '', type: 'bool' }],
  },
];

// ERC20 ABI for USDC approval
export const ERC20_ABI = [
  {
    name: 'approve',
    type: 'function',
    inputs: [
      { name: 'spender', type: 'address' },
      { name: 'amount', type: 'uint256' },
    ],
    outputs: [{ name: '', type: 'bool' }],
  },
  {
    name: 'allowance',
    type: 'function',
    inputs: [
      { name: 'owner', type: 'address' },
      { name: 'spender', type: 'address' },
    ],
    outputs: [{ name: '', type: 'uint256' }],
  },
];

// Output helper
export function output(data, json = false) {
  if (json) {
    console.log(JSON.stringify(data, null, 2));
  } else {
    console.log(data);
  }
}
