#!/usr/bin/env bash
set -euo pipefail

# Agent Identity Kit — Schema Validator
# Usage: ./validate.sh <agent.json> [schema.json]

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
REPO_ROOT="$(cd "$SCRIPT_DIR/../.." && pwd)"

FILE="${1:-}"
SCHEMA="${2:-$REPO_ROOT/schema/agent.schema.json}"

if [ -z "$FILE" ]; then
  echo "Usage: validate.sh <agent.json> [schema.json]"
  echo ""
  echo "Validates an agent.json file against the Agent Card v1 schema."
  exit 1
fi

if [ ! -f "$FILE" ]; then
  echo "❌ File not found: $FILE"
  exit 1
fi

if [ ! -f "$SCHEMA" ]; then
  echo "❌ Schema not found: $SCHEMA"
  echo "   Expected at: $SCHEMA"
  exit 1
fi

# Check for validation tool
if command -v ajv &> /dev/null; then
  echo "🔍 Validating $FILE against Agent Card v1 schema..."
  echo ""
  if ajv validate -s "$SCHEMA" -d "$FILE" --spec=draft7; then
    echo ""
    echo "✅ Valid agent.json!"
  else
    echo ""
    echo "❌ Validation failed. Fix the errors above and try again."
    exit 1
  fi
elif command -v npx &> /dev/null; then
  echo "🔍 Validating $FILE against Agent Card v1 schema..."
  echo "   (Using npx ajv-cli — may take a moment on first run)"
  echo ""
  if npx ajv-cli validate -s "$SCHEMA" -d "$FILE" --spec=draft7; then
    echo ""
    echo "✅ Valid agent.json!"
  else
    echo ""
    echo "❌ Validation failed. Fix the errors above and try again."
    exit 1
  fi
elif command -v python3 &> /dev/null; then
  echo "🔍 Validating $FILE against Agent Card v1 schema..."
  echo ""
  python3 -c "
import json, sys
try:
    from jsonschema import validate, ValidationError
except ImportError:
    print('Installing jsonschema...')
    import subprocess
    subprocess.check_call([sys.executable, '-m', 'pip', 'install', 'jsonschema', '-q'])
    from jsonschema import validate, ValidationError

with open('$SCHEMA') as f:
    schema = json.load(f)
with open('$FILE') as f:
    data = json.load(f)

try:
    validate(instance=data, schema=schema)
    print('✅ Valid agent.json!')
except ValidationError as e:
    print(f'❌ Validation failed: {e.message}')
    print(f'   Path: {\" > \".join(str(p) for p in e.absolute_path)}')
    sys.exit(1)
"
else
  echo "❌ No validator found. Install one of:"
  echo "   npm install -g ajv-cli"
  echo "   pip install jsonschema"
  exit 1
fi
