#!/usr/bin/env bash
set -euo pipefail

# Agent Identity Kit — Interactive agent.json Generator
# Usage: ./init.sh [output_path]

OUTPUT="${1:-agent.json}"

echo "🪪  Agent Identity Kit — Create your agent.json"
echo "================================================"
echo ""

# Agent info
read -rp "Agent name: " AGENT_NAME
read -rp "Handle (@name@domain): " AGENT_HANDLE
read -rp "Description: " AGENT_DESC

# Validate handle format
if [[ ! "$AGENT_HANDLE" =~ ^@[a-zA-Z0-9_-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$ ]]; then
  echo "⚠️  Handle should be in @name@domain format (e.g., @myagent@example.com)"
  echo "   Continuing anyway..."
fi

# Owner info
echo ""
echo "Owner information (who's accountable for this agent):"
read -rp "Owner name: " OWNER_NAME
read -rp "Owner URL (optional): " OWNER_URL
read -rp "Owner contact email (optional): " OWNER_CONTACT

# Capabilities
echo ""
echo "Capabilities (comma-separated, e.g., code-generation,web-search,file-operations):"
read -rp "Capabilities: " CAPS_RAW

# Build capabilities JSON array
CAPS_JSON="[]"
if [ -n "$CAPS_RAW" ]; then
  IFS=',' read -ra CAPS_ARR <<< "$CAPS_RAW"
  CAPS_JSON="["
  FIRST=true
  for cap in "${CAPS_ARR[@]}"; do
    cap=$(echo "$cap" | xargs) # trim whitespace
    if [ "$FIRST" = true ]; then
      FIRST=false
    else
      CAPS_JSON+=","
    fi
    CAPS_JSON+="\"$cap\""
  done
  CAPS_JSON+="]"
fi

# Platform
echo ""
read -rp "Runtime (e.g., openclaw, langchain, custom) [openclaw]: " RUNTIME
RUNTIME="${RUNTIME:-openclaw}"
read -rp "Model (e.g., claude-sonnet-4-20250514) [optional]: " MODEL

# Trust level
echo ""
echo "Trust level: new | active | established | verified"
read -rp "Trust level [new]: " TRUST_LEVEL
TRUST_LEVEL="${TRUST_LEVEL:-new}"

# Timestamps
NOW=$(date -u +"%Y-%m-%dT%H:%M:%SZ")

# Build owner JSON
OWNER_JSON="{\"name\":\"$OWNER_NAME\""
[ -n "$OWNER_URL" ] && OWNER_JSON+=",\"url\":\"$OWNER_URL\""
[ -n "$OWNER_CONTACT" ] && OWNER_JSON+=",\"contact\":\"$OWNER_CONTACT\""
OWNER_JSON+="}"

# Build platform JSON
PLATFORM_JSON="{\"runtime\":\"$RUNTIME\""
[ -n "$MODEL" ] && PLATFORM_JSON+=",\"model\":\"$MODEL\""
PLATFORM_JSON+="}"

# Generate the agent.json
cat > "$OUTPUT" << CARD
{
  "\$schema": "https://foragents.dev/schemas/agent-card/v1.json",
  "version": "1.0",
  "agent": {
    "name": "$AGENT_NAME",
    "handle": "$AGENT_HANDLE",
    "description": "$AGENT_DESC"
  },
  "owner": $OWNER_JSON,
  "platform": $PLATFORM_JSON,
  "capabilities": $CAPS_JSON,
  "protocols": {
    "mcp": false,
    "a2a": false,
    "agent-card": "1.0"
  },
  "trust": {
    "level": "$TRUST_LEVEL",
    "created": "$NOW",
    "verified_by": [],
    "attestations": []
  },
  "created_at": "$NOW",
  "updated_at": "$NOW"
}
CARD

echo ""
echo "✅ Agent card created: $OUTPUT"
echo ""
echo "Next steps:"
echo "  1. Edit $OUTPUT to add endpoints, links, and more capabilities"
echo "  2. Validate: ./scripts/validate.sh $OUTPUT"
echo "  3. Host at: https://yourdomain.com/.well-known/agent.json"
echo "  4. Register at: https://foragents.dev"
