#!/bin/bash
# agent-deploy: Remove an OpenClaw agent
# Usage: bash remove.sh <agentId>

set -euo pipefail

AGENT_ID="${1:?Usage: remove.sh <agentId>}"
CONFIG="$HOME/.openclaw/openclaw.json"
BACKUP="$HOME/.openclaw/openclaw.json.pre-remove-$AGENT_ID"

if [ "$AGENT_ID" = "main" ]; then
    echo "??Cannot remove the main agent!"
    exit 1
fi

echo "?? Agent Remove: $AGENT_ID"

# Backup
cp "$CONFIG" "$BACKUP"
echo "??Backup: $BACKUP"

python3 << PYEOF
import json

config_path = "$CONFIG"
agent_id = "$AGENT_ID"

with open(config_path) as f:
    config = json.load(f)

# Remove from agents.list
agents_list = config.get("agents", {}).get("list", [])
config["agents"]["list"] = [a for a in agents_list if a.get("id") != agent_id]
print(f"??Removed '{agent_id}' from agents.list")

# Remove binding
bindings = config.get("bindings", [])
config["bindings"] = [b for b in bindings if b.get("agentId") != agent_id]
print(f"??Removed binding for '{agent_id}'")

# Remove telegram account
tg = config.get("channels", {}).get("telegram", {})
accounts = tg.get("accounts", {})
if agent_id in accounts:
    del accounts[agent_id]
    print(f"??Removed telegram account '{agent_id}'")

with open(config_path, "w") as f:
    json.dump(config, f, indent=2, ensure_ascii=False)
    f.write("\n")

print(f"??Config updated")
PYEOF

echo ""
echo "???  Workspace NOT deleted: ~/.openclaw/workspace-$AGENT_ID"
echo "   Delete manually if no longer needed."
echo ""
echo "?? Restart: systemctl --user restart openclaw-gateway"
