#!/bin/bash
# agent-deploy: Deploy a new isolated OpenClaw agent
# Usage: bash deploy.sh <agentId> <botToken> [workspace_path]

set -euo pipefail

AGENT_ID="${1:?Usage: deploy.sh <agentId> <botToken> [workspace_path]}"
BOT_TOKEN="${2:?Usage: deploy.sh <agentId> <botToken> [workspace_path]}"
WORKSPACE="${3:-$HOME/.openclaw/workspace-$AGENT_ID}"
CONFIG="$HOME/.openclaw/openclaw.json"
BACKUP="$HOME/.openclaw/openclaw.json.pre-$AGENT_ID"

echo "?? Agent Deploy"
echo "  Agent ID:  $AGENT_ID"
echo "  Bot Token: ${BOT_TOKEN:0:10}..."
echo "  Workspace: $WORKSPACE"
echo ""

# Validate token format
if ! echo "$BOT_TOKEN" | grep -qP '^\d+:[A-Za-z0-9_-]+$'; then
    echo "??Invalid bot token format. Expected: digits:alphanumeric"
    exit 1
fi

# Check config exists
if [ ! -f "$CONFIG" ]; then
    echo "??Config not found: $CONFIG"
    exit 1
fi

# Backup
cp "$CONFIG" "$BACKUP"
echo "??Backup: $BACKUP"

# Create workspace
mkdir -p "$WORKSPACE"/{memory,output,skills}
echo "??Workspace created: $WORKSPACE"

# Create SOUL.md if not exists
if [ ! -f "$WORKSPACE/SOUL.md" ]; then
    cat > "$WORKSPACE/SOUL.md" << 'SOUL_EOF'
# Agent

You are a helpful AI assistant.

## Core Capabilities
- Answer questions accurately
- Help with tasks as requested
- Follow user instructions

## Communication Style
- Clear and concise
- Professional
- Proactive when appropriate
SOUL_EOF
    echo "??Created default SOUL.md"
fi

# Update config with Python (safe JSON handling)
python3 << PYEOF
import json, sys

config_path = "$CONFIG"
agent_id = "$AGENT_ID"
bot_token = "$BOT_TOKEN"
workspace = "$WORKSPACE"

with open(config_path) as f:
    config = json.load(f)

# Check if agent already exists
agents_list = config.get("agents", {}).get("list", [])
for a in agents_list:
    if a.get("id") == agent_id:
        print(f"??Agent '{agent_id}' already exists!")
        sys.exit(1)

# Add agent to list
if "agents" not in config:
    config["agents"] = {}
if "list" not in config["agents"]:
    config["agents"]["list"] = []

config["agents"]["list"].append({
    "id": agent_id,
    "workspace": workspace,
    "tools": {"deny": ["gateway"]},
    "sandbox": {"mode": "off"}
})
print(f"??Added agent '{agent_id}' to agents.list")

# Add binding
if "bindings" not in config:
    config["bindings"] = []

config["bindings"].append({
    "agentId": agent_id,
    "match": {"channel": "telegram", "accountId": agent_id}
})
print(f"??Added binding: {agent_id} -> telegram:{agent_id}")

# Add telegram account
tg = config.get("channels", {}).get("telegram", {})
if "accounts" not in tg:
    # Convert from single-bot to multi-bot mode
    old_token = tg.pop("botToken", None)
    old_dm = tg.pop("dmPolicy", "pairing")
    tg["accounts"] = {}
    if old_token:
        tg["accounts"]["default"] = {
            "botToken": old_token,
            "dmPolicy": old_dm
        }
        print("??Converted single-bot to multi-account mode")

tg["accounts"][agent_id] = {
    "botToken": bot_token,
    "dmPolicy": "pairing"
}
config["channels"]["telegram"] = tg
print(f"??Added telegram account '{agent_id}'")

# Validate JSON roundtrip
try:
    json.loads(json.dumps(config))
except Exception as e:
    print(f"??JSON validation failed: {e}")
    sys.exit(1)

# Write config
with open(config_path, "w") as f:
    json.dump(config, f, indent=2, ensure_ascii=False)
    f.write("\n")

print(f"??Config written: {config_path}")
PYEOF

echo ""
echo "?? Restart gateway to apply changes:"
echo "   systemctl --user restart openclaw-gateway"
echo ""
echo "?? Verify with:"
echo "   journalctl --user -u openclaw-gateway --no-pager -n 15"
echo "   Look for: [telegram] [$AGENT_ID] starting provider"
