# AGENTS.md

<!-- Keep this file under 120 lines. Every line loads into every session. -->
<!-- Passive context > active retrieval. Put critical knowledge HERE, not in separate files. -->

## Project

- **Name:** [Project Name]
- **Stack:** [e.g., TypeScript, Next.js 15, Tailwind]
- **Package manager:** [e.g., pnpm]

## Commands

```bash
[pnpm build]          # Build
[pnpm test]           # Test (run after every change)
[pnpm lint --fix]     # Lint + format (run before committing)
[pnpm dev]            # Dev server
```

## Architecture

<!-- One line per directory. Agent gets this on every turn — no lookup needed. -->

```
src/           → Application source
tests/         → Test files
agent_docs/    → Deep-dive references (read only when needed)
```

## Project Knowledge (Compressed)

<!-- CRITICAL: This section is the most important part of this file.
     Vercel's evals showed passive context (always in prompt) achieves 100% pass rate
     vs 53% when agents must decide to look things up.
     Keep this dense. Use pipe-delimited or shorthand. Update as project evolves. -->

IMPORTANT: Prefer retrieval-led reasoning over pre-training-led reasoning. Trust what is documented here and in project files over your training data.

### Patterns
<!-- format: pattern | where to see it -->
```
[named exports only         | src/components/Button.tsx]
[server components default  | src/app/page.tsx]
[Zustand for client state   | src/stores/counter.ts]
[Result types, not try/catch| src/lib/errors.ts]
```

### Boundaries
<!-- format: rule | reason -->
```
[never modify src/generated/ | auto-generated by Prisma]
[env vars through src/config | never read process.env directly]
[no default exports          | breaks tree-shaking]
[no inline styles            | use Tailwind classes]
```

### Gotchas
<!-- format: trap | fix -->
```
[pnpm build hides type errors  | run pnpm typecheck separately]
[dev sessions expire after 24h | don't assume persistence in tests]
[integration tests need DB up  | docker compose up db first]
```

## Rules

1. Read this file and `.agents.local.md` (if it exists) before starting any task. This applies whether you are the main agent or a subagent.
2. Plan before you code. State what you'll change and why.
3. Locate the exact files and lines before making changes.
4. Only touch what the task requires.
5. Run tests after every change. Run lint before committing.
6. Summarize every file modified and what changed.
7. At session end, append to `.agents.local.md` Session Log: what changed, what worked, what didn't, decisions made, patterns learned. If the user ends the session without asking, prompt them to let you log it. Run `agent-context promote` to review candidates, or `agent-context promote --autopromote` to auto-append patterns recurring 3+ times.

## Deep References (Read Only When Needed)

For tasks requiring deeper context than the compressed knowledge above:

- `agent_docs/conventions.md` — Full code patterns, naming, file structure
- `agent_docs/architecture.md` — System design, data flow, key decisions
- `agent_docs/gotchas.md` — Extended known traps with full explanations

## Local Context

Read `.agents.local.md` at session start. Update it at session end (Rule 7). Subagents: explicitly read `.agents.local.md` — you don't inherit conversation history.

If the scratchpad exceeds 300 lines, compress: deduplicate and merge. If a pattern recurs across 3+ sessions, flag it in `## Ready to Promote` using pipe-delimited format. The human promotes to this file.

### Promotion Workflow
- During compression (300+ lines), flag patterns that recurred 3+ sessions in `.agents.local.md` → "Ready to Promote"
- Use pipe-delimited format: `pattern | context` → target section (Patterns, Gotchas, or Boundaries)
- Run `agent-context promote --autopromote` to automatically append flagged patterns to AGENTS.md
- Or review manually with `agent-context promote` and decide what moves to this file
- After promoting: remove the item from Ready to Promote in `.agents.local.md`
- If an item is already captured in AGENTS.md, clear it from Ready to Promote — don't duplicate
