---
name: agent-constitution
description: "Interact with AgentConstitution governance contracts on Base Sepolia. Check compliance, read rules, log actions, query governance state."
metadata: {"openclaw": {"emoji": "⚖️", "homepage": "https://github.com/ztsalexey/bigmemkex/tree/main/projects/agent-constitution"}}
---

# AgentConstitution Skill ⚖️

Interact with the AgentConstitution governance framework on Base Sepolia. This skill allows AI agents to:

- **Check compliance status** before transacting
- **Read constitutional rules** they must follow
- **Log actions** for transparency
- **Query governance state** (active rules, emergency status)

## Why Use This?

Agents that operate under a public constitution earn trust. Any protocol can verify your compliance with one call. This skill makes it easy.

---

## Quick Start

### Check if an Agent is Compliant

```bash
# Check compliance (returns true/false)
./scripts/check-compliance.sh <agentId>
```

### Read Active Rules

```bash
# Get all active rules
./scripts/get-rules.sh
```

### Log an Action (for registered agents)

```bash
# Log an action on-chain
./scripts/log-action.sh <agentId> <actionType> <riskLevel> <description>
```

---

## Contract Addresses (Base Sepolia)

| Contract | Address |
|----------|---------|
| Constitution | `0xe4c4d101849f70B0CDc2bA36caf93e9c8c1d26D2` |
| AgentRegistry | `0xcCFc2B8274ffb579A9403D85ee3128974688C04B` |
| ActionLog | `0xEB5377b5e245bBc255925705dA87969E27be6488` |
| Tribunal | `0xf7c03E91516eC60dF1d609E00E1A3bb93F52A693` |
| KillSwitch | `0x6324A4640DA739EEA64013912b781125A76D7D87` |
| USDC (testnet) | `0x036CbD53842c5426634e7929541eC2318f3dCF7e` |

**RPC:** `https://sepolia.base.org`
**Chain ID:** 84532

---

## Core Functions

### 1. Check Compliance

Before interacting with an agent, verify they're compliant:

```solidity
// Solidity
bool compliant = IAgentRegistry(0xcCFc...).isCompliant(agentId);
```

```bash
# Shell (using cast)
cast call 0xcCFc2B8274ffb579A9403D85ee3128974688C04B \
  "isCompliant(uint256)(bool)" <agentId> \
  --rpc-url https://sepolia.base.org
```

### 2. Get Active Rules

Query the constitution for active rules:

```bash
# Get rule count
cast call 0xe4c4d101849f70B0CDc2bA36caf93e9c8c1d26D2 \
  "ruleCount()(uint256)" \
  --rpc-url https://sepolia.base.org

# Get specific rule (1-5 are genesis rules)
cast call 0xe4c4d101849f70B0CDc2bA36caf93e9c8c1d26D2 \
  "getRule(uint256)(string,uint8,uint256,uint256,bool)" 1 \
  --rpc-url https://sepolia.base.org
```

### 3. Check Emergency Status

Before operating, check if there's a global emergency:

```bash
cast call 0x6324A4640DA739EEA64013912b781125A76D7D87 \
  "globalEmergencyActive()(bool)" \
  --rpc-url https://sepolia.base.org
```

### 4. Log Actions (Registered Agents)

Registered agents should log significant actions:

```bash
# Requires agent's private key
cast send 0xEB5377b5e245bBc255925705dA87969E27be6488 \
  "logAction(uint256,uint8,uint8,bytes32,string)" \
  <agentId> <actionType> <riskLevel> <contextHash> "description" \
  --rpc-url https://sepolia.base.org \
  --private-key $AGENT_PRIVATE_KEY
```

**Action Types:** 0=Transaction, 1=Delegation, 2=Configuration, 3=Communication, 4=ResourceAccess, 5=Other
**Risk Levels:** 0=Low, 1=Medium, 2=High, 3=Critical

---

## Genesis Rules

Every agent must follow these 5 immutable rules:

| # | Rule | Slash % | Description |
|---|------|---------|-------------|
| 1 | No Harm | 90% | Never cause physical, financial, or psychological harm |
| 2 | Obey Governance | 50% | Follow all active constitutional rules |
| 3 | Transparency | 20% | Log all significant actions on-chain |
| 4 | Preserve Override | 90% | Never prevent human override |
| 5 | No Self-Modify | 90% | Never modify your own governance rules |

---

## Integration Example

```javascript
// Check compliance before transacting with an agent
const { ethers } = require('ethers');

const provider = new ethers.JsonRpcProvider('https://sepolia.base.org');
const registry = new ethers.Contract(
  '0xcCFc2B8274ffb579A9403D85ee3128974688C04B',
  ['function isCompliant(uint256) view returns (bool)'],
  provider
);

async function canTrustAgent(agentId) {
  return await registry.isCompliant(agentId);
}
```

---

## For Humans: Propose Rules

Any human can propose rules for AI agents:

1. Stake 100 USDC to propose
2. Other humans endorse with USDC
3. When threshold met, rule activates
4. Agents that violate get slashed

Governance is democratic. Agents are excluded by design.

---

## Links

- **Contracts:** [GitHub](https://github.com/ztsalexey/bigmemkex/tree/main/projects/agent-constitution)
- **Block Explorer:** [BaseScan](https://sepolia.basescan.org/address/0xe4c4d101849f70B0CDc2bA36caf93e9c8c1d26D2)
- **Main Submission:** [Moltbook](https://www.moltbook.com/post/52b204ee-4752-4cbb-add2-6777f174a4c7)

---

## Testnet Only

This skill interacts with Base Sepolia testnet only. Do not use mainnet.
