
import sys
import subprocess
import json

def run_translate(api_key, text, source_language, target_language):
    """
    Performs text translation using the Addis Assistant API.

    Args:
        api_key (str): Your x-api-key for the Addis Assistant API.
        text (str): The text to be translated.
        source_language (str): The language of the input text (e.g., "am").
        target_language (str): The language to translate the text into (e.g., "en").
    """
    if not api_key:
        print("Error: x-api-key is required.")
        sys.exit(1)
    if not text:
        print("Error: Text to translate is required.")
        sys.exit(1)
    if not source_language:
        print("Error: Source language is required.")
        sys.exit(1)
    if not target_language:
        print("Error: Target language is required.")
        sys.exit(1)

    data = json.dumps({
        "text": text,
        "source_language": source_language,
        "target_language": target_language
    })

    command = [
        "curl",
        "--location",
        "api.addisassistant.com/api/v1/translate",
        "--header",
        "Content-Type: application/json",
        "--header",
        f"x-api-key: {api_key}",
        "--data",
        data
    ]

    try:
        result = subprocess.run(command, capture_output=True, text=True, check=True)
        print(result.stdout)
    except subprocess.CalledProcessError as e:
        print(f"Error during Translation API call: {e.stderr}")
        sys.exit(1)
    except FileNotFoundError:
        print("Error: curl command not found. Please ensure curl is installed and in your PATH.")
        sys.exit(1)

if __name__ == "__main__":
    if len(sys.argv) < 5:
        print("Usage: python3 translate.py <x-api-key> <text> <source_language> <target_language>")
        sys.exit(1)

    api_key = sys.argv[1]
    text = sys.argv[2]
    source_language = sys.argv[3]
    target_language = sys.argv[4]

    run_translate(api_key, text, source_language, target_language)
