# Adaptive Learning Agent

**Learn from errors and corrections in real-time. Continuously improve by capturing failures, user feedback, and successful patterns.**

Free and open-source (MIT License) • Zero dependencies • Works locally

---

## 🚀 Why This Skill?

### Problem Statement
Working with Claude or any AI agent means encountering:
- Mistakes that need correction
- Unexpected API behaviors
- Better approaches discovered through experimentation
- Knowledge gaps that get revealed during use

But there's no systematic way to **learn from these moments** and apply the knowledge next time.

### The Solution
**Adaptive Learning Agent** captures every error, correction, and successful pattern automatically. Then retrieves relevant learnings before tackling similar problems again.

### Real Use Cases
- **Bug discovery**: Record an error once, never struggle with it again
- **Prompt optimization**: Keep track of what prompt variations work best
- **API integration**: Remember quirky behaviors and workarounds
- **Workflow improvement**: Document shortcuts and best practices
- **Team knowledge**: Export and share learnings across projects

---

## ✨ What You Get

### Four Core Functions

**1. Record Learnings**
```python
agent.record_learning(
    content="Use claude-sonnet for 90% of tasks—faster and cheaper",
    category="technique",
    context="Model selection"
)
```
Capture successful patterns, insights, and best practices.

**2. Record Errors**
```python
agent.record_error(
    error_description="JSON parsing failed on null values",
    context="Processing API response",
    solution="Add null check before parsing"
)
```
Document failures and solutions automatically.

**3. Search & Retrieve Learnings**
```python
results = agent.search_learnings("JSON parsing")
recent = agent.get_recent_learnings(limit=5)
by_category = agent.get_learnings_by_category("bug-fix")
```
Find relevant knowledge instantly when you need it.

**4. View Summaries**
```python
summary = agent.get_learning_summary()
print(agent.format_learning_summary())
```
Understand what you've learned at a glance.

### Key Features

✅ **Zero dependencies** - Pure Python, works everywhere
✅ **Local-only storage** - All data on your machine, no uploads
✅ **MIT Licensed** - Free to use, modify, fork, redistribute
✅ **Automatic categorization** - Errors become learnings
✅ **Search and filter** - Find knowledge by keyword or category
✅ **Export capability** - Share learnings as JSON
✅ **No API keys** - Works without any external credentials

---

## 📊 Real-World Example

```python
from adaptive_learning_agent import AdaptiveLearningAgent

# Initialize agent
agent = AdaptiveLearningAgent()

# Day 1: Discover a bug
agent.record_error(
    error_description="Anthropic API rejects prompts with excessive newlines",
    context="Testing prompt with formatted lists",
    solution="Use \\n.strip() to clean whitespace before sending"
)

# Day 2: Same bug, but now you have the solution
similar_errors = agent.search_learnings("newlines")
# Result: [Previous learning with solution] ✅

# Week 1: Document successful pattern
agent.record_learning(
    content="Always use temperature=0 for deterministic output in tests",
    category="best-practice",
    context="Prompt engineering"
)

# Get weekly summary
summary = agent.get_learning_summary()
print(f"You've recorded {summary['total_learnings']} learnings this week!")
print(f"Resolved {summary['error_statistics']['resolved']} errors")
```

---

## 🔧 Installation

No installation needed! The skill is pure Python with zero dependencies.

```bash
# Copy the adaptive_learning_agent.py file to your project
# Or import it directly:

from adaptive_learning_agent import AdaptiveLearningAgent
```

---

## 💡 Use Cases

### Software Development
Record bugs you find and their fixes. Next time you hit a similar error, you have the solution ready.

```python
agent.record_error(
    error_description="Port 8000 already in use",
    context="Running local dev server",
    solution="Use `lsof -i :8000` to find process, then kill it"
)
```

### Prompt Engineering
Keep track of prompting techniques that work for your specific use cases.

```python
agent.record_learning(
    content="Chain-of-thought works better for math problems, direct answers for facts",
    category="technique"
)
```

### API Integration
Remember quirky behaviors and workarounds for each provider.

```python
agent.record_learning(
    content="OpenAI API requires explicit 'assistant' role messages",
    category="api-endpoint",
    context="Chat completion endpoint"
)
```

### Team Knowledge
Export learnings and share with your team or future projects.

```python
agent.export_learnings("team_learnings.json")
# Share this file with teammates
```

### Continuous Improvement
Before major tasks, review what you've learned to avoid repeating mistakes.

```python
summary = agent.get_learning_summary()
unresolved = summary['error_statistics']['unresolved']
if unresolved > 0:
    print(f"⚠️ {unresolved} unresolved errors—review before proceeding")
```

---

## 📚 Categories

When recording learnings, choose from these categories:

| Category | Use For |
|----------|---------|
| **technique** | Working methods, approaches, strategies |
| **bug-fix** | Solutions to errors and problems |
| **api-endpoint** | API-specific behaviors and quirks |
| **constraint** | Limits, boundaries, restrictions |
| **best-practice** | Recommended patterns and standards |
| **error-handling** | How to handle specific types of errors |

---

## 🎯 Sources

When recording learnings, specify the source:

- `user-correction` - User told you something was wrong
- `error-discovery` - You found the solution to an error
- `successful-pattern` - You discovered something that works well
- `user-feedback` - User suggested an improvement

---

## 📖 API Reference

### Core Methods

#### `record_learning(content, category, source, context)`
Record a successful pattern or insight.

**Parameters:**
- `content` (str, required): What was learned
- `category` (str): One of the category types above
- `source` (str): One of the source types above
- `context` (str): Optional context about where this applies

**Returns:** `Learning` object with ID and timestamp

#### `record_error(error_description, context, solution, prevention_tip)`
Record an error and optionally its solution.

**Parameters:**
- `error_description` (str, required): What went wrong
- `context` (str, required): What was being attempted
- `solution` (str): How to fix it
- `prevention_tip` (str): How to avoid it

**Returns:** `Error` object with ID

#### `search_learnings(query)`
Search learnings by keyword or category.

**Parameters:**
- `query` (str): Search term

**Returns:** List of matching `Learning` objects (sorted by relevance)

#### `get_recent_learnings(limit)`
Get the most recent learnings.

**Parameters:**
- `limit` (int): Number to return (default: 10)

**Returns:** List of `Learning` objects, newest first

#### `get_learning_summary()`
Get comprehensive summary of learnings and errors.

**Returns:** Dictionary with statistics and recent items

#### `export_learnings(output_file)`
Export all learnings and errors to JSON file.

**Parameters:**
- `output_file` (str): Path to save JSON (default: "learnings_export.json")

---

## 🔒 Privacy & Security

- ✅ **Zero telemetry** - No data sent anywhere
- ✅ **Local-only storage** - Everything stored in `.adaptive_learning/` on your machine
- ✅ **No API calls** - Works completely offline
- ✅ **No authentication** - No accounts, keys, or logins needed
- ✅ **Full transparency** - Source code included and open-source

---

## 🤝 Contributing

This is MIT Licensed and community-maintained. You're encouraged to:
- Fork the repository
- Submit improvements and features
- Integrate it into your projects
- Share learnings with others

---

## 📝 Changelog

### [1.0.0] - 2026-02-14

#### ✨ Initial Release
- **Core learning system** - Record and retrieve learnings
- **Error tracking** - Capture errors with solutions
- **Search functionality** - Find learnings by keyword or category
- **Local storage** - All data stays on your machine
- **Export capability** - Share learnings as JSON files
- **Zero dependencies** - Pure Python, no external packages
- **MIT Licensed** - Free to use, modify, redistribute
- **Comprehensive API** - Simple, Pythonic interface

---

## 📞 Support

- **GitHub**: https://github.com/clawhub-skills/adaptive-learning-agent
- **Issues & Contributions**: Open an issue or PR on GitHub
- **Community**: Share your learnings and improvements!

---

## 📄 License

**MIT License** - Free and open-source

Use, modify, fork, and redistribute freely. See [LICENSE.md](LICENSE.md) for full details.

```
Copyright © 2026 UnisAI Community
```

---

**Last Updated**: February 14, 2026
**Current Version**: 1.0.0
**Status**: Active & Community-Maintained

Free to use, modify, and fork. No restrictions.
