#!/usr/bin/env node
// Watch for new bids on the ABN network
// Usage: node src/watch.js [industry]

import { SimplePool } from 'nostr-tools/pool';
import { nip19 } from 'nostr-tools';
import { RELAYS, KINDS, isRelatedIndustry } from './config.js';

async function watchBids(filters = {}) {
  const pool = new SimplePool();
  
  const filter = {
    kinds: [KINDS.LINK_BID],
    '#t': ['abn-bid'],
    since: Math.floor(Date.now() / 1000) - 86400 // Last 24 hours
  };
  
  if (filters.industry) {
    filter['#t'].push(filters.industry);
  }
  
  if (filters.type) {
    filter['#t'].push(filters.type);
  }
  
  console.log('Watching for ABN bids...');
  console.log('Filter:', JSON.stringify(filter, null, 2));
  console.log('\nPress Ctrl+C to stop.\n');
  
  const sub = pool.subscribeMany(RELAYS, [filter], {
    onevent(event) {
      const bid = JSON.parse(event.content);
      const npub = nip19.npubEncode(event.pubkey);
      
      console.log('━'.repeat(50));
      console.log(`⚡ New ${bid.type.toUpperCase()} bid`);
      console.log(`   From: ${npub.slice(0, 20)}...`);
      console.log(`   Industry: ${bid.industry}`);
      
      if (bid.type === 'seeking') {
        console.log(`   Target: ${bid.targetSite}`);
        console.log(`   Wants: DA${bid.requirements?.minDA}+ ${bid.requirements?.linkType}`);
        console.log(`   Paying: ${bid.offer?.sats} sats`);
      } else {
        console.log(`   Site: ${bid.site}`);
        console.log(`   DA: ${bid.da}`);
        console.log(`   Placement: ${bid.placement}`);
        console.log(`   Price: ${bid.price?.sats} sats`);
      }
      
      // Check expiry
      const expiryTag = event.tags.find(t => t[0] === 'expiry');
      if (expiryTag) {
        const expiry = new Date(parseInt(expiryTag[1]) * 1000);
        console.log(`   Expires: ${expiry.toLocaleDateString()}`);
      }
      
      console.log(`   Event: ${event.id.slice(0, 16)}...`);
      console.log('');
    },
    oneose() {
      console.log('(Caught up with historical events, now watching live...)\n');
    }
  });
  
  // Keep running
  process.on('SIGINT', () => {
    console.log('\nStopping...');
    sub.close();
    pool.close(RELAYS);
    process.exit(0);
  });
}

// Also query existing bids
async function queryBids(filters = {}) {
  const pool = new SimplePool();
  
  const filter = {
    kinds: [KINDS.LINK_BID],
    '#t': ['abn-bid'],
    '#l': ['link-bid']
  };
  
  if (filters.industry) {
    filter['#t'].push(filters.industry);
  }
  
  console.log('Querying existing bids...\n');
  
  const events = await pool.querySync(RELAYS, filter);
  const now = Math.floor(Date.now() / 1000);
  
  // Filter out expired bids
  const activeBids = events.filter(e => {
    const expiryTag = e.tags.find(t => t[0] === 'expiry');
    if (!expiryTag) return true;
    return parseInt(expiryTag[1]) > now;
  });
  
  console.log(`Found ${activeBids.length} active bids:\n`);
  
  for (const event of activeBids) {
    const bid = JSON.parse(event.content);
    const npub = nip19.npubEncode(event.pubkey);
    
    if (bid.type === 'seeking') {
      console.log(`🔍 SEEKING: ${bid.industry} links for ${bid.targetSite}`);
      console.log(`   Wants DA${bid.requirements?.minDA}+, paying ${bid.offer?.sats} sats`);
    } else {
      console.log(`📢 OFFERING: ${bid.linkType} on ${bid.site} (DA${bid.da})`);
      console.log(`   ${bid.placement} placement, ${bid.price?.sats} sats`);
    }
    console.log(`   Contact: ${npub.slice(0, 30)}...`);
    console.log('');
  }
  
  pool.close(RELAYS);
  return activeBids;
}

// CLI - only run when executed directly
const isMainModule = process.argv[1]?.endsWith('watch.js');
if (isMainModule) {
  const [,, industry] = process.argv;
  
  queryBids({ industry }).then(() => {
    console.log('\nNow watching for new bids...\n');
    watchBids({ industry });
  });
}

export { watchBids, queryBids };
