#!/usr/bin/env node
// ABN Link Verification Module
// Crawls pages to verify that backlinks were actually placed
// Usage: node src/verify.js <url> <target-domain>

import https from 'https';
import http from 'http';

/**
 * Fetch a URL and return HTML content
 * @param {string} url - URL to fetch
 * @returns {Promise<string>} - HTML content
 */
async function fetchPage(url, options = {}) {
  return new Promise((resolve, reject) => {
    const protocol = url.startsWith('https') ? https : http;
    const timeout = options.timeout || 10000;
    
    const req = protocol.get(url, {
      headers: {
        'User-Agent': 'ABN-LinkVerifier/1.0 (Agent Backlink Network)',
        'Accept': 'text/html,application/xhtml+xml',
        ...options.headers
      },
      timeout
    }, (res) => {
      // Handle redirects
      if (res.statusCode >= 300 && res.statusCode < 400 && res.headers.location) {
        const redirectUrl = res.headers.location.startsWith('http') 
          ? res.headers.location 
          : new URL(res.headers.location, url).href;
        return fetchPage(redirectUrl, options).then(resolve).catch(reject);
      }
      
      if (res.statusCode !== 200) {
        reject(new Error(`HTTP ${res.statusCode}`));
        return;
      }
      
      let data = '';
      res.on('data', chunk => data += chunk);
      res.on('end', () => resolve(data));
    });
    
    req.on('error', reject);
    req.on('timeout', () => {
      req.destroy();
      reject(new Error('Request timed out'));
    });
  });
}

/**
 * Extract all links from HTML
 * @param {string} html - HTML content
 * @returns {array} - Array of link objects { href, anchor, rel }
 */
function extractLinks(html) {
  const links = [];
  const linkRegex = /<a\s+([^>]*href\s*=\s*["']([^"']+)["'][^>]*)>([^<]*)<\/a>/gi;
  
  let match;
  while ((match = linkRegex.exec(html)) !== null) {
    const [, attrs, href, anchor] = match;
    const rel = attrs.match(/rel\s*=\s*["']([^"']+)["']/i)?.[1] || '';
    links.push({
      href: href.trim(),
      anchor: anchor.trim(),
      rel: rel.toLowerCase(),
      isDoFollow: !rel.includes('nofollow'),
      isSponsored: rel.includes('sponsored'),
      isUGC: rel.includes('ugc')
    });
  }
  
  return links;
}

/**
 * Verify a backlink exists on a page
 * @param {string} pageUrl - URL of page to check
 * @param {string} targetDomain - Domain to look for in links
 * @param {object} options - { anchor, dofollow, exactUrl }
 * @returns {Promise<object>} - Verification result
 */
async function verifyBacklink(pageUrl, targetDomain, options = {}) {
  try {
    const html = await fetchPage(pageUrl);
    const links = extractLinks(html);
    
    // Find links pointing to target domain
    const matchingLinks = links.filter(link => {
      try {
        const url = new URL(link.href, pageUrl);
        return url.hostname.includes(targetDomain) || link.href.includes(targetDomain);
      } catch {
        return link.href.includes(targetDomain);
      }
    });
    
    if (matchingLinks.length === 0) {
      return {
        verified: false,
        pageUrl,
        targetDomain,
        message: 'No links found to target domain',
        checkedAt: new Date().toISOString()
      };
    }
    
    let bestMatch = matchingLinks[0];
    
    if (options.dofollow) {
      const dofollowLinks = matchingLinks.filter(l => l.isDoFollow);
      if (dofollowLinks.length === 0) {
        return {
          verified: false,
          pageUrl,
          targetDomain,
          linksFound: matchingLinks,
          message: 'Links found but all are nofollow',
          checkedAt: new Date().toISOString()
        };
      }
      bestMatch = dofollowLinks[0];
    }
    
    if (options.anchor && !matchingLinks.some(l => 
      l.anchor.toLowerCase().includes(options.anchor.toLowerCase()))) {
      return {
        verified: false,
        pageUrl,
        targetDomain,
        linksFound: matchingLinks,
        message: `Links found but anchor text "${options.anchor}" not matched`,
        checkedAt: new Date().toISOString()
      };
    }
    
    if (options.exactUrl && !matchingLinks.some(l => l.href === options.exactUrl)) {
      return {
        verified: false,
        pageUrl,
        targetDomain,
        linksFound: matchingLinks,
        message: `Links found but exact URL "${options.exactUrl}" not matched`,
        checkedAt: new Date().toISOString()
      };
    }
    
    return {
      verified: true,
      pageUrl,
      targetDomain,
      linksFound: matchingLinks,
      bestMatch,
      message: `Found ${matchingLinks.length} link(s) to ${targetDomain}`,
      checkedAt: new Date().toISOString()
    };
    
  } catch (err) {
    return {
      verified: false,
      pageUrl,
      targetDomain,
      error: err.message,
      message: `Failed to fetch page: ${err.message}`,
      checkedAt: new Date().toISOString()
    };
  }
}

/**
 * Batch verify multiple backlinks
 * @param {array} checks - Array of { pageUrl, targetDomain, options }
 * @returns {Promise<array>} - Array of verification results
 */
async function batchVerify(checks) {
  const results = [];
  for (const check of checks) {
    const result = await verifyBacklink(check.pageUrl, check.targetDomain, check.options);
    results.push(result);
    await new Promise(r => setTimeout(r, 500));
  }
  return results;
}

/**
 * Generate a verification report
 * @param {array} results - Array of verification results
 * @returns {string} - Formatted report
 */
function generateReport(results) {
  const verified = results.filter(r => r.verified);
  const failed = results.filter(r => !r.verified);
  
  let report = '═'.repeat(60) + '\n';
  report += '           ABN BACKLINK VERIFICATION REPORT\n';
  report += '═'.repeat(60) + '\n\n';
  report += `Total Checked: ${results.length}\n`;
  report += `Verified: ${verified.length} ✓\n`;
  report += `Failed: ${failed.length} ✗\n\n`;
  
  if (verified.length > 0) {
    report += '─'.repeat(60) + '\n';
    report += 'VERIFIED LINKS:\n';
    report += '─'.repeat(60) + '\n';
    for (const r of verified) {
      report += `✓ ${r.pageUrl}\n`;
      report += `  → ${r.targetDomain}\n`;
      if (r.bestMatch) {
        report += `  Anchor: "${r.bestMatch.anchor}"\n`;
        report += `  Type: ${r.bestMatch.isDoFollow ? 'dofollow' : 'nofollow'}\n`;
      }
      report += '\n';
    }
  }
  
  if (failed.length > 0) {
    report += '─'.repeat(60) + '\n';
    report += 'FAILED VERIFICATIONS:\n';
    report += '─'.repeat(60) + '\n';
    for (const r of failed) {
      report += `✗ ${r.pageUrl}\n`;
      report += `  → ${r.targetDomain}\n`;
      report += `  Reason: ${r.message}\n`;
      if (r.linksFound?.length > 0) {
        report += `  Found: ${r.linksFound.map(l => l.href).join(', ')}\n`;
      }
      report += '\n';
    }
  }
  
  report += '═'.repeat(60) + '\n';
  report += `Report generated: ${new Date().toISOString()}\n`;
  
  return report;
}

// CLI - only run when executed directly
const isMainModule = process.argv[1]?.endsWith('verify.js');
if (isMainModule) {
  const [,, pageUrl, targetDomain, ...flags] = process.argv;
  
  if (!pageUrl || !targetDomain) {
    console.log('ABN Link Verification');
    console.log('Usage: node src/verify.js <page-url> <target-domain> [flags]');
    console.log('');
    console.log('Flags:');
    console.log('  --dofollow         Only accept dofollow links');
    console.log('  --anchor="text"    Require specific anchor text');
    console.log('  --exact="url"      Require exact URL match');
    console.log('');
    console.log('Examples:');
    console.log('  node src/verify.js https://example.com/partners acmeplumbing.com');
    console.log('  node src/verify.js https://blog.com/post mydomain.com --dofollow');
    process.exit(0);
  }
  
  const options = {};
  for (const flag of flags) {
    if (flag === '--dofollow') options.dofollow = true;
    if (flag.startsWith('--anchor=')) options.anchor = flag.split('=')[1].replace(/"/g, '');
    if (flag.startsWith('--exact=')) options.exactUrl = flag.split('=')[1].replace(/"/g, '');
  }
  
  console.log(`🔍 Checking ${pageUrl} for links to ${targetDomain}...`);
  verifyBacklink(pageUrl, targetDomain, options).then(result => {
    console.log('\n' + generateReport([result]));
  }).catch(console.error);
}

export { verifyBacklink, batchVerify, extractLinks, fetchPage, generateReport };
