/**
 * Test script for Agent Backlink Network
 * 
 * Run: npm run test
 */

import { webcrypto } from 'node:crypto';
// @ts-ignore - polyfill for Node < 20
if (!globalThis.crypto) globalThis.crypto = webcrypto;

import 'websocket-polyfill';
import { NostrClient, parseSiteEvent, DEFAULT_RELAYS } from './lib/nostr.js';
import { verifyLink } from './lib/verifier.js';
import { loadState, addSite, getSites, getStatePath } from './lib/state.js';

async function test() {
  console.log('🧪 Agent Backlink Network - Test Suite\n');
  
  // Test 1: State management
  console.log('1️⃣ Testing state management...');
  const state = loadState();
  console.log(`   ✓ State loaded from ${getStatePath()}`);
  console.log(`   ✓ Public key: ${state.npub.slice(0, 20)}...`);
  
  // Test 2: Local site storage
  console.log('\n2️⃣ Testing local site storage...');
  const testSite = {
    url: 'https://test-plumber.example.com',
    businessName: 'Test Plumber Co',
    businessType: 'plumber',
    location: {
      city: 'San Diego',
      state: 'CA',
      country: 'US',
      radiusMiles: 25,
    },
    linkPages: ['/partners', '/local-resources'],
    lookingFor: ['hvac', 'electrician', 'roofer'],
  };
  addSite(testSite);
  const sites = getSites();
  console.log(`   ✓ Site added: ${testSite.businessName}`);
  console.log(`   ✓ Total sites: ${sites.length}`);

  // Test 3: Link verification
  console.log('\n3️⃣ Testing link verification...');
  try {
    // Test with a real page that we know has links
    const result = await verifyLink('https://example.com', 'https://www.iana.org');
    console.log(`   ✓ Verification ran successfully`);
    console.log(`   ✓ Link found: ${result.found}`);
    if (result.anchorText) console.log(`   ✓ Anchor text: ${result.anchorText}`);
  } catch (e: any) {
    console.log(`   ⚠ Verification test skipped: ${e.message}`);
  }

  // Test 4: Nostr connection
  console.log('\n4️⃣ Testing Nostr connection...');
  const client = new NostrClient(state.privateKey, DEFAULT_RELAYS.slice(0, 2));
  console.log(`   ✓ Client created`);
  console.log(`   ✓ Client npub: ${client.npub}`);

  try {
    // Try to find some sites on the network
    console.log('   ⏳ Querying network for registered sites...');
    const events = await Promise.race([
      client.findSites({}),
      new Promise<never>((_, reject) => 
        setTimeout(() => reject(new Error('timeout')), 10000)
      ),
    ]);
    console.log(`   ✓ Found ${events.length} sites on the network`);
    
    if (events.length > 0) {
      // Find first valid site
      for (const event of events) {
        const site = parseSiteEvent(event);
        if (site) {
          console.log(`   ✓ Sample site: ${site.businessName} (${site.businessType})`);
          break;
        }
      }
    }
  } catch (e: any) {
    if (e.message === 'timeout') {
      console.log('   ⚠ Query timed out (normal if no sites registered yet)');
    } else {
      console.log(`   ⚠ Query failed: ${e.message}`);
    }
  }

  // Test 5: Event creation (don't publish)
  console.log('\n5️⃣ Testing event creation...');
  console.log('   ⏳ Creating test registration event...');
  
  // We'll just verify event creation works without publishing
  const { finalizeEvent } = await import('nostr-tools');
  const testEvent = finalizeEvent({
    kind: 30100,
    created_at: Math.floor(Date.now() / 1000),
    tags: [
      ['d', testSite.url],
      ['url', testSite.url],
      ['name', testSite.businessName],
      ['type', testSite.businessType],
    ],
    content: JSON.stringify(testSite),
  }, hexToBytes(state.privateKey));
  
  console.log(`   ✓ Event created with ID: ${testEvent.id.slice(0, 16)}...`);
  console.log(`   ✓ Event signature valid: ${testEvent.sig.slice(0, 16)}...`);

  client.close();

  console.log('\n✅ All tests passed!\n');
  console.log('Next steps:');
  console.log('  1. Register a real site: npm run dev -- register-site -u https://yoursite.com -n "Your Business" -t plumber -c "San Diego" -s CA');
  console.log('  2. Find matches: npm run dev -- find-matches');
  console.log('  3. Check your status: npm run dev -- status');
}

function hexToBytes(hex: string): Uint8Array {
  const bytes = new Uint8Array(hex.length / 2);
  for (let i = 0; i < hex.length; i += 2) {
    bytes[i / 2] = parseInt(hex.substr(i, 2), 16);
  }
  return bytes;
}

test().catch(console.error);
