#!/usr/bin/env node
// Register a site to the ABN network
// Usage: NOSTR_PRIVATE_KEY=nsec1... node src/register.js

import { finalizeEvent, getPublicKey, nip19 } from 'nostr-tools';
import { Relay } from 'nostr-tools/relay';
import { RELAYS, KINDS, loadPrivateKey } from './config.js';

// Parse private key
function parsePrivateKey(key) {
  if (key.startsWith('nsec')) {
    const decoded = nip19.decode(key);
    return decoded.data;
  }
  // Assume hex
  return new Uint8Array(key.match(/.{1,2}/g).map(b => parseInt(b, 16)));
}

async function registerSite(site) {
  const privKey = parsePrivateKey(loadPrivateKey());
  const pubkey = getPublicKey(privKey);
  
  console.log('Registering site:', site.name);
  console.log('Your pubkey:', nip19.npubEncode(pubkey));
  
  const event = finalizeEvent({
    kind: KINDS.SITE_REGISTRATION,
    created_at: Math.floor(Date.now() / 1000),
    tags: [
      ['d', site.url],
      ['t', 'abn-site'],
      ['t', site.industry],
      ['t', site.type || 'local-business'],
      ['L', 'abn'],
      ['l', 'site-registration', 'abn']
    ],
    content: JSON.stringify({
      name: site.name,
      url: site.url,
      city: site.city,
      state: site.state,
      industry: site.industry,
      da: site.da || null,
      wantLinks: site.wantLinks || ['homepage'],
      canOffer: site.canOffer || ['footer', 'partners-page'],
      registeredAt: new Date().toISOString()
    })
  }, privKey);
  
  console.log('Event ID:', event.id);
  console.log('Publishing to relays...');
  
  for (const url of RELAYS) {
    try {
      const relay = await Relay.connect(url);
      await relay.publish(event);
      console.log(`✓ ${url}`);
      relay.close();
    } catch (err) {
      console.log(`✗ ${url}: ${err.message}`);
    }
  }
  
  console.log('\nDone! Your site is now on the ABN network.');
}

// Example usage - modify for your site
const mySite = {
  name: 'Example Plumbing Co',
  url: 'https://example-plumbing.com',
  city: 'San Diego',
  state: 'CA',
  industry: 'plumbing',
  type: 'local-business',
  da: 25,
  wantLinks: ['homepage', 'service-pages'],
  canOffer: ['footer', 'partners-page', 'blog']
};

// Uncomment to run:
// registerSite(mySite);

export { registerSite };
