#!/usr/bin/env node
// Query the ABN network for sites
// Usage: node src/query.js [industry] [state]

import { SimplePool } from 'nostr-tools/pool';
import { nip19 } from 'nostr-tools';
import { RELAYS, KINDS, isRelatedIndustry } from './config.js';

async function querySites(filters = {}) {
  const pool = new SimplePool();
  
  const filter = {
    kinds: [KINDS.SITE_REGISTRATION],
    '#t': ['abn-site'],
    '#l': ['site-registration']
  };
  
  if (filters.industry) {
    filter['#t'].push(filters.industry);
  }
  
  console.log('Querying ABN network...');
  console.log('Filter:', JSON.stringify(filter, null, 2));
  
  const events = await pool.querySync(RELAYS, filter);
  
  console.log(`\nFound ${events.length} sites:\n`);
  
  const sites = events.map(e => {
    const data = JSON.parse(e.content);
    return {
      pubkey: e.pubkey,
      npub: nip19.npubEncode(e.pubkey),
      eventId: e.id,
      ...data
    };
  });
  
  // Filter by state if specified
  let filtered = sites;
  if (filters.state) {
    filtered = sites.filter(s => s.state === filters.state);
  }
  
  // Sort by DA
  filtered.sort((a, b) => (b.da || 0) - (a.da || 0));
  
  for (const site of filtered) {
    console.log(`📍 ${site.name}`);
    console.log(`   URL: ${site.url}`);
    console.log(`   Location: ${site.city}, ${site.state}`);
    console.log(`   Industry: ${site.industry}`);
    console.log(`   DA: ${site.da || 'unknown'}`);
    console.log(`   Pubkey: ${site.npub.slice(0, 20)}...`);
    console.log('');
  }
  
  pool.close(RELAYS);
  return filtered;
}

function findMatches(yourSite, allSites) {
  return allSites
    .filter(s => s.url !== yourSite.url) // Not yourself
    .map(site => {
      let score = 0;
      
      // Same state = good for local relevance
      if (yourSite.state === site.state) score += 30;
      
      // Different city = no direct competition
      if (yourSite.city !== site.city) score += 20;
      
      // Industry relevance
      if (yourSite.industry === site.industry) score += 25;
      else if (isRelatedIndustry(yourSite.industry, site.industry)) score += 20;
      
      // DA bonus
      if (site.da >= 30) score += 15;
      if (site.da >= 40) score += 10;
      
      return { ...site, matchScore: score };
    })
    .filter(s => s.matchScore > 30) // Minimum threshold
    .sort((a, b) => b.matchScore - a.matchScore);
}

// CLI usage - only run when executed directly
const isMainModule = process.argv[1]?.endsWith('query.js');
if (isMainModule) {
  const [,, industry, state] = process.argv;
  
  if (industry || state) {
    querySites({ industry, state });
  } else {
    console.log('Usage: node src/query.js [industry] [state]');
    console.log('Example: node src/query.js plumbing CA');
    console.log('\nRunning without filters...\n');
    querySites({});
  }
}

export { querySites, findMatches };
