#!/usr/bin/env node
/**
 * ABN Lightning Payment Module
 * 
 * This module provides helpers for Lightning payments in ABN deals.
 * It's designed to work with common Lightning backends:
 * - LNbits (recommended for agents)
 * - Alby API
 * - BTCPay Server
 * - LND (direct)
 * 
 * Usage: Configure your Lightning backend in .secrets/lightning.json
 */

import { readFileSync, existsSync } from 'fs';
import { join } from 'path';
import { fileURLToPath } from 'url';
import { dirname } from 'path';

const __dirname = dirname(fileURLToPath(import.meta.url));
const projectRoot = join(__dirname, '..');

// Load Lightning config
function loadConfig() {
  const configPath = join(projectRoot, '.secrets', 'lightning.json');
  if (!existsSync(configPath)) {
    return null;
  }
  return JSON.parse(readFileSync(configPath, 'utf-8'));
}

/**
 * LNbits API Client
 * Free to use, easy to set up: https://lnbits.com
 */
class LNbitsClient {
  constructor(baseUrl, apiKey) {
    this.baseUrl = baseUrl.replace(/\/$/, '');
    this.apiKey = apiKey;
  }
  
  async createInvoice(amount, memo = 'ABN Link Payment') {
    const response = await fetch(`${this.baseUrl}/api/v1/payments`, {
      method: 'POST',
      headers: {
        'X-Api-Key': this.apiKey,
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        out: false,
        amount: amount, // sats
        memo,
        unit: 'sat'
      })
    });
    
    if (!response.ok) {
      throw new Error(`LNbits error: ${response.status}`);
    }
    
    const data = await response.json();
    return {
      paymentRequest: data.payment_request,
      paymentHash: data.payment_hash,
      amount,
      memo
    };
  }
  
  async payInvoice(bolt11) {
    const response = await fetch(`${this.baseUrl}/api/v1/payments`, {
      method: 'POST',
      headers: {
        'X-Api-Key': this.apiKey,
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        out: true,
        bolt11
      })
    });
    
    if (!response.ok) {
      throw new Error(`LNbits error: ${response.status}`);
    }
    
    const data = await response.json();
    return {
      paymentHash: data.payment_hash,
      preimage: data.checking_id
    };
  }
  
  async checkPayment(paymentHash) {
    const response = await fetch(`${this.baseUrl}/api/v1/payments/${paymentHash}`, {
      headers: {
        'X-Api-Key': this.apiKey
      }
    });
    
    if (!response.ok) {
      throw new Error(`LNbits error: ${response.status}`);
    }
    
    const data = await response.json();
    return {
      paid: data.paid,
      preimage: data.preimage,
      amount: data.amount
    };
  }
  
  async getBalance() {
    const response = await fetch(`${this.baseUrl}/api/v1/wallet`, {
      headers: {
        'X-Api-Key': this.apiKey
      }
    });
    
    if (!response.ok) {
      throw new Error(`LNbits error: ${response.status}`);
    }
    
    const data = await response.json();
    return {
      balance: Math.floor(data.balance / 1000), // msats to sats
      name: data.name
    };
  }
}

/**
 * Get a Lightning client based on config
 */
function getClient() {
  const config = loadConfig();
  if (!config) {
    console.log(`
╔══════════════════════════════════════════════════════════╗
║           LIGHTNING NOT CONFIGURED                        ║
╚══════════════════════════════════════════════════════════╝

To enable Lightning payments, create .secrets/lightning.json:

For LNbits (recommended):
{
  "provider": "lnbits",
  "baseUrl": "https://legend.lnbits.com",
  "apiKey": "your-wallet-invoice-read-key"
}

For Alby:
{
  "provider": "alby",
  "apiKey": "your-alby-bearer-token"
}

Free LNbits wallets: https://legend.lnbits.com
Alby: https://getalby.com
`);
    return null;
  }
  
  switch (config.provider) {
    case 'lnbits':
      return new LNbitsClient(config.baseUrl, config.apiKey);
    default:
      throw new Error(`Unknown Lightning provider: ${config.provider}`);
  }
}

/**
 * Create an invoice for an ABN deal
 * @param {number} sats - Amount in satoshis
 * @param {string} dealId - ABN deal/bid ID for reference
 * @returns {Promise<object>} - Invoice details
 */
async function createInvoice(sats, dealId = 'abn-deal') {
  const client = getClient();
  if (!client) {
    throw new Error('Lightning not configured');
  }
  
  return client.createInvoice(sats, `ABN Link Exchange: ${dealId}`);
}

/**
 * Pay a Lightning invoice
 * @param {string} bolt11 - BOLT11 invoice string
 * @returns {Promise<object>} - Payment result with preimage
 */
async function payInvoice(bolt11) {
  const client = getClient();
  if (!client) {
    throw new Error('Lightning not configured');
  }
  
  return client.payInvoice(bolt11);
}

/**
 * Check if a payment was received
 * @param {string} paymentHash - Payment hash to check
 * @returns {Promise<object>} - Payment status
 */
async function checkPayment(paymentHash) {
  const client = getClient();
  if (!client) {
    throw new Error('Lightning not configured');
  }
  
  return client.checkPayment(paymentHash);
}

/**
 * Get wallet balance
 * @returns {Promise<object>} - Balance in sats
 */
async function getBalance() {
  const client = getClient();
  if (!client) {
    throw new Error('Lightning not configured');
  }
  
  return client.getBalance();
}

// CLI usage
const [,, action, ...args] = process.argv;

async function main() {
  switch (action) {
    case 'balance':
      const balance = await getBalance();
      console.log(`💰 Wallet: ${balance.name}`);
      console.log(`   Balance: ${balance.balance.toLocaleString()} sats`);
      break;
      
    case 'invoice':
      const sats = parseInt(args[0]) || 1000;
      const dealId = args[1] || `deal-${Date.now()}`;
      const invoice = await createInvoice(sats, dealId);
      console.log(`\n⚡ Invoice created for ${sats} sats`);
      console.log(`\nPayment Request (BOLT11):\n${invoice.paymentRequest}`);
      console.log(`\nPayment Hash: ${invoice.paymentHash}`);
      break;
      
    case 'pay':
      if (!args[0]) {
        console.log('Usage: node src/lightning.js pay <bolt11-invoice>');
        process.exit(1);
      }
      console.log('Paying invoice...');
      const result = await payInvoice(args[0]);
      console.log(`✓ Payment sent!`);
      console.log(`  Hash: ${result.paymentHash}`);
      console.log(`  Preimage: ${result.preimage}`);
      break;
      
    case 'check':
      if (!args[0]) {
        console.log('Usage: node src/lightning.js check <payment-hash>');
        process.exit(1);
      }
      const status = await checkPayment(args[0]);
      console.log(`Payment status: ${status.paid ? '✓ PAID' : '⏳ PENDING'}`);
      if (status.preimage) console.log(`Preimage: ${status.preimage}`);
      break;
      
    default:
      console.log('ABN Lightning Payment Module');
      console.log('Usage: node src/lightning.js <action> [args]');
      console.log('');
      console.log('Actions:');
      console.log('  balance              - Check wallet balance');
      console.log('  invoice <sats> [id]  - Create invoice');
      console.log('  pay <bolt11>         - Pay an invoice');
      console.log('  check <hash>         - Check payment status');
      console.log('');
      
      // Show config status
      const config = loadConfig();
      if (config) {
        console.log(`\n✓ Lightning configured: ${config.provider}`);
      } else {
        console.log('\n⚠️  Lightning not configured (see above for setup)');
      }
  }
}

// Only run CLI when executed directly
const isMainModule = process.argv[1]?.endsWith('lightning.js');
if (isMainModule) {
  main().catch(err => {
    console.error('Error:', err.message);
    process.exit(1);
  });
}

export { createInvoice, payInvoice, checkPayment, getBalance, getClient };
