#!/usr/bin/env node
// ABN Encrypted DM Module for agent negotiation (NIP-04)
// Usage: node src/dm.js <action> [args]

import { finalizeEvent, getPublicKey, nip19, nip04 } from 'nostr-tools';
import { SimplePool } from 'nostr-tools/pool';
import { Relay } from 'nostr-tools/relay';
import { RELAYS, KINDS, loadPrivateKey } from './config.js';

function parsePrivateKey(key) {
  if (key.startsWith('nsec')) {
    const decoded = nip19.decode(key);
    return decoded.data;
  }
  return new Uint8Array(key.match(/.{1,2}/g).map(b => parseInt(b, 16)));
}

function parsePublicKey(key) {
  if (key.startsWith('npub')) {
    const decoded = nip19.decode(key);
    return decoded.data;
  }
  return key; // Assume hex
}

/**
 * Send an encrypted DM to another agent
 * @param {string} recipientPubkey - npub or hex pubkey
 * @param {object} message - Message object to send
 * @returns {Promise<object>} - Event object
 */
async function sendDM(recipientPubkey, message) {
  const privKey = parsePrivateKey(loadPrivateKey());
  const pubkey = getPublicKey(privKey);
  const recipientHex = parsePublicKey(recipientPubkey);
  
  // Encrypt the message content
  const plaintext = JSON.stringify(message);
  const ciphertext = await nip04.encrypt(privKey, recipientHex, plaintext);
  
  const event = finalizeEvent({
    kind: KINDS.ENCRYPTED_DM,
    created_at: Math.floor(Date.now() / 1000),
    tags: [['p', recipientHex]],
    content: ciphertext
  }, privKey);
  
  console.log('Sending encrypted DM...');
  console.log(`From: ${nip19.npubEncode(pubkey).slice(0, 20)}...`);
  console.log(`To: ${recipientPubkey.slice(0, 20)}...`);
  console.log(`Type: ${message.type}`);
  
  let success = 0;
  for (const url of RELAYS.slice(0, 3)) { // Use first 3 relays for DMs
    try {
      const relay = await Relay.connect(url);
      await relay.publish(event);
      console.log(`✓ ${url}`);
      success++;
      relay.close();
    } catch (err) {
      console.log(`✗ ${url}: ${err.message}`);
    }
  }
  
  if (success === 0) {
    throw new Error('Failed to publish to any relay');
  }
  
  return event;
}

/**
 * Read DMs sent to your pubkey
 * @param {object} options - { since: unix_timestamp, from: npub }
 * @returns {Promise<array>} - Decrypted messages
 */
async function readDMs(options = {}) {
  const privKey = parsePrivateKey(loadPrivateKey());
  const pubkey = getPublicKey(privKey);
  
  const pool = new SimplePool();
  
  const filter = {
    kinds: [KINDS.ENCRYPTED_DM],
    '#p': [pubkey],
    since: options.since || Math.floor(Date.now() / 1000) - 86400 * 7 // Last 7 days
  };
  
  if (options.from) {
    filter.authors = [parsePublicKey(options.from)];
  }
  
  console.log('Fetching encrypted DMs...');
  
  const events = await pool.querySync(RELAYS.slice(0, 3), filter);
  
  const messages = [];
  for (const event of events) {
    try {
      const plaintext = await nip04.decrypt(privKey, event.pubkey, event.content);
      const parsed = JSON.parse(plaintext);
      messages.push({
        id: event.id,
        from: nip19.npubEncode(event.pubkey),
        fromHex: event.pubkey,
        timestamp: event.created_at,
        date: new Date(event.created_at * 1000).toISOString(),
        ...parsed
      });
    } catch (err) {
      // Skip messages we can't decrypt (not for us or malformed)
      console.log(`Skipping message ${event.id.slice(0, 8)}...: ${err.message}`);
    }
  }
  
  // Sort by timestamp, newest first
  messages.sort((a, b) => b.timestamp - a.timestamp);
  
  pool.close(RELAYS);
  return messages;
}

/**
 * Subscribe to live DMs
 * @param {function} callback - Called with each new decrypted message
 */
async function watchDMs(callback) {
  const privKey = parsePrivateKey(loadPrivateKey());
  const pubkey = getPublicKey(privKey);
  
  const pool = new SimplePool();
  
  console.log(`Watching for DMs to ${nip19.npubEncode(pubkey).slice(0, 20)}...`);
  console.log('Press Ctrl+C to stop.\n');
  
  const sub = pool.subscribeMany(RELAYS.slice(0, 3), [{
    kinds: [KINDS.ENCRYPTED_DM],
    '#p': [pubkey],
    since: Math.floor(Date.now() / 1000)
  }], {
    async onevent(event) {
      try {
        const plaintext = await nip04.decrypt(privKey, event.pubkey, event.content);
        const parsed = JSON.parse(plaintext);
        const message = {
          id: event.id,
          from: nip19.npubEncode(event.pubkey),
          fromHex: event.pubkey,
          timestamp: event.created_at,
          ...parsed
        };
        callback(message);
      } catch (err) {
        // Skip
      }
    }
  });
  
  process.on('SIGINT', () => {
    console.log('\nStopping...');
    sub.close();
    pool.close(RELAYS);
    process.exit(0);
  });
}

// ABN-specific message types
const MessageTypes = {
  // Initial inquiry about a bid/site
  inquiry: (bidId, message) => ({
    type: 'inquiry',
    regarding: bidId,
    message,
    timestamp: new Date().toISOString()
  }),
  
  // Counter-offer
  counter: (sats, terms) => ({
    type: 'counter',
    sats,
    terms,
    timestamp: new Date().toISOString()
  }),
  
  // Accept deal
  accept: (invoice) => ({
    type: 'accept',
    invoice,
    timestamp: new Date().toISOString()
  }),
  
  // Confirm payment
  paid: (preimage, linkDetails) => ({
    type: 'paid',
    preimage,
    linkDetails, // { url, anchor }
    timestamp: new Date().toISOString()
  }),
  
  // Confirm link placed
  placed: (liveUrl, proof) => ({
    type: 'placed',
    liveUrl,
    proof, // URL to screenshot or hash
    timestamp: new Date().toISOString()
  }),
  
  // Verify and close deal
  verified: (confirmed, notes) => ({
    type: 'verified',
    confirmed,
    notes,
    timestamp: new Date().toISOString()
  })
};

// CLI usage
const [,, action, ...args] = process.argv;

async function main() {
  switch (action) {
    case 'read':
      const messages = await readDMs({ since: args[0] ? parseInt(args[0]) : undefined });
      console.log(`\nFound ${messages.length} messages:\n`);
      for (const msg of messages) {
        console.log(`📨 ${msg.type.toUpperCase()} from ${msg.from.slice(0, 20)}...`);
        console.log(`   Date: ${msg.date}`);
        console.log(`   Content: ${JSON.stringify(msg, null, 2).slice(0, 200)}...`);
        console.log('');
      }
      break;
      
    case 'send':
      if (args.length < 2) {
        console.log('Usage: node src/dm.js send <npub> <type> [args...]');
        console.log('Types: inquiry, counter, accept, paid, placed, verified');
        process.exit(1);
      }
      const [npub, type, ...msgArgs] = args;
      let msg;
      switch (type) {
        case 'inquiry':
          msg = MessageTypes.inquiry(msgArgs[0] || 'general', msgArgs[1] || 'Interested in link exchange');
          break;
        case 'counter':
          msg = MessageTypes.counter(parseInt(msgArgs[0]) || 5000, msgArgs[1] || 'Standard terms');
          break;
        default:
          console.log('Unknown message type. Use: inquiry, counter, accept, paid, placed, verified');
          process.exit(1);
      }
      await sendDM(npub, msg);
      break;
      
    case 'watch':
      await watchDMs((msg) => {
        console.log('━'.repeat(50));
        console.log(`📬 New ${msg.type.toUpperCase()} from ${msg.from.slice(0, 20)}...`);
        console.log(JSON.stringify(msg, null, 2));
        console.log('');
      });
      break;
      
    default:
      console.log('ABN Encrypted DM Module');
      console.log('Usage: node src/dm.js <action> [args]');
      console.log('');
      console.log('Actions:');
      console.log('  read [since]       - Read recent DMs');
      console.log('  send <npub> <type> - Send a DM');
      console.log('  watch              - Watch for new DMs');
      console.log('');
      console.log('Message types: inquiry, counter, accept, paid, placed, verified');
  }
}

// Only run CLI when executed directly
const isMainModule = process.argv[1]?.endsWith('dm.js');
if (isMainModule) {
  main().catch(console.error);
}

export { sendDM, readDMs, watchDMs, MessageTypes };
