#!/usr/bin/env node
// Post a link bid to the ABN network
// Usage: NOSTR_PRIVATE_KEY=nsec1... node src/bid.js

import { finalizeEvent, getPublicKey, nip19 } from 'nostr-tools';
import { Relay } from 'nostr-tools/relay';
import { RELAYS, KINDS, loadPrivateKey } from './config.js';

function parsePrivateKey(key) {
  if (key.startsWith('nsec')) {
    const decoded = nip19.decode(key);
    return decoded.data;
  }
  return new Uint8Array(key.match(/.{1,2}/g).map(b => parseInt(b, 16)));
}

async function postBid(bid) {
  const privKey = parsePrivateKey(loadPrivateKey());
  const pubkey = getPublicKey(privKey);
  
  const bidId = `bid-${Date.now()}`;
  const expiryDays = bid.expiryDays || 7;
  const expiry = Math.floor(Date.now() / 1000) + (86400 * expiryDays);
  
  console.log(`Posting ${bid.type} bid...`);
  console.log('Your pubkey:', nip19.npubEncode(pubkey));
  
  const event = finalizeEvent({
    kind: KINDS.LINK_BID,
    created_at: Math.floor(Date.now() / 1000),
    tags: [
      ['d', bidId],
      ['t', 'abn-bid'],
      ['t', bid.type], // 'seeking' or 'offering'
      ['t', bid.industry],
      ['L', 'abn'],
      ['l', 'link-bid', 'abn'],
      ['amount', String(bid.sats)],
      ['expiry', String(expiry)]
    ],
    content: JSON.stringify(bid)
  }, privKey);
  
  console.log('Bid ID:', bidId);
  console.log('Event ID:', event.id);
  console.log('Expires:', new Date(expiry * 1000).toISOString());
  console.log('Publishing to relays...');
  
  for (const url of RELAYS) {
    try {
      const relay = await Relay.connect(url);
      await relay.publish(event);
      console.log(`✓ ${url}`);
      relay.close();
    } catch (err) {
      console.log(`✗ ${url}: ${err.message}`);
    }
  }
  
  console.log('\nBid posted! Other agents can now see and respond to it.');
}

// Example: Seeking links
const seekingBid = {
  type: 'seeking',
  targetSite: 'https://acmeplumbing.com',
  industry: 'plumbing',
  requirements: {
    minDA: 25,
    industries: ['plumbing', 'hvac', 'construction'],
    states: ['CA', 'AZ', 'NV'],
    linkType: 'dofollow',
    placement: ['content', 'resource-page', 'partners']
  },
  offer: {
    sats: 5000,
    paymentTerms: 'on-verification'
  },
  expiryDays: 7
};

// Example: Offering links
const offeringBid = {
  type: 'offering',
  site: 'https://bestroofing.com',
  industry: 'roofing',
  da: 35,
  placement: 'partners-page',
  linkType: 'dofollow',
  price: {
    sats: 3000,
    paymentTerms: 'upfront'
  },
  restrictions: {
    industries: ['roofing', 'construction', 'home-improvement'],
    noCompetitors: true,
    maxLinks: 5
  },
  expiryDays: 30
};

// Uncomment to run:
// postBid(seekingBid);
// postBid(offeringBid);

export { postBid };
